! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and two port valve
! *              - calculates water-side flow rate from inlet pressure
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. ww      : water mass flow rate                            (kg/s)
! *  7. qa      : total heat transfer to the air                    (kW)
! *  8. shr     : sensible heat ratio                                (-)
! *  9. effect  : coil effectiveness                                 (-)
! * 10. bf      : coil by-pass factor                                (-)
! * 11. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 12. rho     : outlet air relative humidity                       (-)
! * 13. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : heigth of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 19. eqpchar : valve curvature parameter (0=linear)               (-)
! * 20. sv      : valve rangability                                  (-)
! * 21. cl      : valve leakage (fractional flow)                    (-)
!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! * 22. xlin1   : stem position at which second segment starts       (-)
! * 23. xlin2   : stem position at which third segment starts        (-)
! * 24. frac2   : flow fraction at third segment start position      (-)
!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    Shokouh Pourarian, Jin Wen
!                        July 13, 2012
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type520(xin,yout,par_v,saved_v,iostat,chcoilfault,vchcoilfault)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=13,np=24,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

!******************************Drexel added**********************************
		integer,dimension(2),intent(in)   :: chcoilfault
		real,dimension(3),intent(in)      :: vchcoilfault
!****************************************************************************

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)



        real         :: xlin=0.01,xeqp=0.33 
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,ww1,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,dpw,wqudlin,rlinport3
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir

!******************************Drexel added**********************************
        real         :: xlin1,xlin2,frac2
		integer      :: VLVcoil,coilfoul
 		real         :: VVLVcoil,Vcoilfoul1,Vcoilfoul2

! **** Read in fault types
        VLVcoil    = chcoilfault(1)
		coilfoul   = chcoilfault(2)

! **** Read in fault severity 
        VVLVcoil   = vchcoilfault(1)
	    Vcoilfoul1 = vchcoilfault(2)
	    Vcoilfoul2 = vchcoilfault(3)
!****************************************************************************

 ! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        pwi     = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
! *** Fault flag on************Drexel added**********************************
		if (VLVcoil.EQ.1) then
! *** Cooling coil valve is stuck
			vstem = VVLVcoil
		endif
! *** Fault flag off*********************************************************
        tsdyn   = xin(9)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)

! *** Fault flag on************Drexel added*********************************
	    if (coilfoul.EQ.2) then
! *** Coil water side fouling let water resistance increased
			frwcoil = Vcoilfoul1*par_v(17)
		endif
! *** Fault flag off********************************************************

! **** Valve characteristics
        kv      = par_v(18)
        eqpchar = par_v(19)
        sv      = par_v(20)
        cl      = par_v(21)

!******************************Drexel added**********************************
		xlin1   = par_v(22)
		xlin2   = par_v(23)
		frac2   = par_v(24)
!****************************************************************************

! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)

! *** Fault flag on************Drexel added********************************
		if (coilfoul.EQ.1) then
! *** Coil air side fouling let heat transfer rate decreased
			confin  = Vcoilfoul2*confin
			contube = Vcoilfoul2*contube

	     elseif (coilfoul.EQ.2) then
! *** Coil water side fouling let heat transfer rate decreased
			confin  = Vcoilfoul2*confin
			contube = Vcoilfoul2*contube
		 endif  
! *** Fault flag off********************************************************

        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
! **********************Drexel Added*****************************
!            frwvalve = rlinport(vstem,kv,sv,xlin,cl)!***turned to comment*********
            frwvalve = rlinport3(vstem,kv,sv,xlin1,xlin2,cl,frac2)
! *********************************************************************************
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frwvalve = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
! **** Total water-side resistance - coil + valve
        frwtot  = frwcoil+frwvalve
        if (frwtot<frsmall) then
! **** Total resistance almost zero
            write(*,*)&
            'type520: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Calculate air inlet pressures and water flow rate from flow resistances
        pai = pao+dpqudlin(fra,wa)
        dpw = pwi-pwo
        ww1  = wqudlin(frwtot,dpw)
! **** Fault flag on************Drexel addes***********************************
        if (VLVcoil.EQ.0) then
! *** Coil valve is fault free
			ww = ww1
        elseif (VLVcoil.EQ.1) then
! *** Coil valve is stuck
			ww = ww1
        elseif (VLVcoil.EQ.2) then
! *** Cooling valve is leaking
            if (vstem.EQ.0) then
                ww = VVLVcoil
            elseif (vstem.NE.0.and.ww.LE.VVLVcoil)then
			    ww = VVLVcoil
			else
				ww = ww1
            endif
	    endif
! **** Fault flag off***********************************************************

! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.					 
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0

                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = ww
        yout(7)  = qa
        yout(8)  = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(9)  = -99.9
        else
            yout(9)  = effect
        endif
        yout(10) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(11) = ho
            yout(12) = rho/100.
            yout(13) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(11) = -99.9
            yout(12) = -99.9
            yout(13) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type520
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and two port valve
! *              - calculates water-side flow rate from inlet pressure
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *  6. ww      : inlet water flow rate                           (kg/s)
! *  7. tsdyn   : effective coil surface temperature                 (C)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!deleted!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!deleted!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *  6. ww      : water mass flow rate                            (kg/s)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *  7. qa      : total heat transfer to the air                    (kW)
! *  8. shr     : sensible heat ratio                                (-)
! *  9. effect  : coil effectiveness                                 (-)
! * 10. bf      : coil by-pass factor                                (-)
! * 11. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 12. rho     : outlet air relative humidity                       (-)
! * 13. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : heigth of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 19. eqpchar : valve curvature parameter (0=linear)               (-)
! * 20. sv      : valve rangability                                  (-)
! * 21. cl      : valve leakage (fractional flow)                    (-)
!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! * 22. xlin1   : stem position at which second segment starts       (-)
! * 23. xlin2   : stem position at which third segment starts        (-)
! * 24. frac2   : flow fraction at third segment start position      (-)
!!!!!!!!!!!!!!!!!!!!!!!Drexel added!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    Shokouh Pourarian, Jin Wen
!                        July 13, 2012
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type478(xin,yout,par_v,saved_v,iostat,chcoilfault,vchcoilfault)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=7,no=12,np=24,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

!******************************Drexel added**********************************
		integer,dimension(2),intent(in)   :: chcoilfault
		real,dimension(3),intent(in)      :: vchcoilfault
!****************************************************************************

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)



        real         :: xlin=0.01,xeqp=0.33 
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,ww1,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,dpw,wqudlin,rlinport3
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir

!******************************Drexel added**********************************
        real         :: xlin1,xlin2,frac2
		integer      :: VLVcoil,coilfoul
 		real         :: VVLVcoil,Vcoilfoul1,Vcoilfoul2

! **** Read in fault types
        VLVcoil    = chcoilfault(1)
		coilfoul   = chcoilfault(2)

! **** Read in fault severity 
        VVLVcoil   = vchcoilfault(1)
	    Vcoilfoul1 = vchcoilfault(2)
	    Vcoilfoul2 = vchcoilfault(3)
!****************************************************************************

 ! **** Read in inputs
!        xin(1)  = xin(1)-4.0
        tai     = xin(1)
!		tai     = tai+5.5556
!		xin(1)  = tai
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
		ww      = xin(6)
       tsdyn   = xin(7)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)

! *** Fault flag on************Drexel added*********************************
	    if (coilfoul.EQ.2) then
! *** Coil water side fouling let water resistance increased
			frwcoil = Vcoilfoul1*par_v(17)
		endif
! *** Fault flag off********************************************************

! **** Valve characteristics
        kv      = par_v(18)
        eqpchar = par_v(19)
        sv      = par_v(20)
        cl      = par_v(21)

!******************************Drexel added**********************************
		xlin1   = par_v(22)
		xlin2   = par_v(23)
		frac2   = par_v(24)
!****************************************************************************

! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)

! *** Fault flag on************Drexel added********************************
		if (coilfoul.EQ.1) then
! *** Coil air side fouling let heat transfer rate decreased
			confin  = Vcoilfoul2*confin
			contube = Vcoilfoul2*contube

	     elseif (coilfoul.EQ.2) then
! *** Coil water side fouling let heat transfer rate decreased
			confin  = Vcoilfoul2*confin
			contube = Vcoilfoul2*contube
		 endif  
! *** Fault flag off********************************************************

        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
!        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
!        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
! **********************Shokouh changed for new valve*****************************
!            frwvalve = rlinport(vstem,kv,sv,xlin,cl)!***turned to comment*********
!            frwvalve = rlinport3(vstem,kv,sv,xlin1,xlin2,cl,frac2)
! *********************************************************************************
!        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
!            frwvalve = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
!        endif
! **** Total water-side resistance - coil + valve
!        frwtot  = frwcoil+frwvalve
!        if (frwtot<frsmall) then
! **** Total resistance almost zero
!            write(*,*)&
!            'type520: water-side flow resistance must not be < ',frsmall
!            stop
!        endif
! **** Calculate air inlet pressures and water flow rate from flow resistances
        pai = pao+dpqudlin(fra,wa)

! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.					 
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
!			ww  = ww + 0.035
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
!        yout(6)  = ww
        yout(6)  = qa
        yout(7)  = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(8)  = -99.9
        else
            yout(8)  = effect
        endif
        yout(9) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(10) = ho
            yout(11) = rho/100.
            yout(12) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(10) = -99.9
            yout(11) = -99.9
            yout(12) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type478
! **********************************************************************
! Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and two port valve
! *              - calculates water-side inlet pressure from flow rate
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. ww      : water mass flow rate                            (kg/s)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. qa      : total heat transfer to the air                    (kW)
! *  8. shr     : sensible heat ratio                                (-)
! *  9. effect  : coil effectiveness                                 (-)
! * 10. bf      : coil by-pass factor                                (-)
! * 11. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 12. rho     : outlet air relative humidity                       (-)
! * 13. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : heigth of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 19. eqpchar : valve curvature parameter (0=linear)               (-)
! * 20. sv      : valve rangability                                  (-)
! * 21. cl      : valve leakage (fractional flow)                    (-)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * VOLCPM  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * CONMA   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type521(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=13,np=21,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir

! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        ww      = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
        tsdyn   = xin(9)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
! **** Valve characteristics
        kv      = par_v(18)
        eqpchar = par_v(19)
        sv      = par_v(20)
        cl      = par_v(21)
! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx      = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
            frwvalve=rlinport(vstem,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frwvalve=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
! **** Total water-side resistance - coil + valve
        frwtot  = frwcoil+frwvalve
        if (frwtot<frsmall) then
! **** Total resistance almost zero
            write(*,*)&
            'type521: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Calculate air and water inlet pressures from flow resistances
        pai     = pao+dpqudlin(fra,wa)
        pwi     = pwo+dpqudlin(frwtot,ww)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = pwi
        yout(7)  = qa
        yout(8)  = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(9)  = -99.9
        else
            yout(9)  = effect
        endif
        yout(10) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(11) = ho
            yout(12) = rho/100.
            yout(13) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(11) = -99.9
            yout(12) = -99.9
            yout(13) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type521

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and two port valve
! *              - calculates water-side flow rate from inlet pressure
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. ww      : water mass flow rate                            (kg/s)
! *  7. qa      : total heat transfer to the air                    (kW)
! *  8. shr     : sensible heat ratio                                (-)
! *  9. effect  : coil effectiveness                                 (-)
! * 10. bf      : coil by-pass factor                                (-)
! * 11. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 12. rho     : outlet air relative humidity                       (-)
! * 13. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : heigth of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 19. eqpchar : valve curvature parameter (0=linear)               (-)
! * 20. sv      : valve rangability                                  (-)
! * 21. cl      : valve leakage (fractional flow)                    (-)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type522(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=13,np=21,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,dpw,wqudlin
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir

 ! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        pwi     = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
        tsdyn   = xin(9)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
! **** Valve characteristics
        kv      = par_v(18)
        eqpchar = par_v(19)
        sv      = par_v(20)
        cl      = par_v(21)
! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve type and resistance
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
! **** Select valve type and calculate resistance
        if (eqpchar<1.0) then
! **** Linear characteristic - two segments (linear and close-off)
            frwvalve = rlinport(vstem,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frwvalve = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
! **** Total water-side resistance - coil + valve
        frwtot  = frwcoil+frwvalve
        if (frwtot<frsmall) then
! **** Total resistance almost zero
            write(*,*)&
            'type522: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Calculate air and water inlet pressures from flow resistances
        pai = pao+dpqudlin(fra,wa)
        dpw = pwi-pwo
        ww  = wqudlin(frwtot,dpw)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = pwi
        yout(7)  = qa
        yout(8)  = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(9)  = -99.9
        else
            yout(9)  = effect
        endif
        yout(10) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(11) = ho
            yout(12) = rho/100.
            yout(13) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(11) = -99.9
            yout(12) = -99.9
            yout(13) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type522
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and three port valve
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil with a three port valve.
! *              Calculates inlet pressure and coil water flow from
! *              primary circuit flow rate and outlet pressure
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. wwprim  : primary circuit water mass flow rate            (kg/s)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. ww      : water mass flow rate though coil                (kg/s)
! *  8. tret    : mixed return water temperature                     (C)
! *  9. qa      : total heat transfer to the air                    (kW)
! * 10. shr     : sensible heat ratio                                (-)
! * 11. effect  : coil effectiveness                                 (-)
! * 12. bf      : coil by-pass factor                                (-)
! * 13. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 14. rho     : outlet air relative humidity                       (-)
! * 15. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : height of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. frwbypas: by-pass water flow resistance             (0.001 kg.m)
! * 19. ivaltype: valve type: 0=lin/lin, 1=eq%(flow)/lin(byp), 2=lin/eq%
! * 20. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 21. eqpchar : valve curvature parameter (equal percentage port)  (-)
! * 22. sv      : valve rangability                                  (-)
! * 23. cl      : valve leakage (fractional flow)                    (-)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type523(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=15,np=23,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,wwprim,frwbypas,&
                        vstembyp,frvalflo,frvalbyp,frtotflo,frtotbyp,&
                        frcombin,dp,dpqud,wqud,tret
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir,ivaltype

! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        wwprim  = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
        tsdyn   = xin(9)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
        frwbypas= par_v(18)
! **** Valve characteristics
        ivaltype= par_v(19)
        kv      = par_v(20)
        eqpchar = par_v(21)
        sv      = par_v(22)
        cl      = par_v(23)
! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve types and resistances
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
        vstembyp= 1.-vstem
! **** Flow port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==2) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalflo=rlinport(vstem,kv,sv,xlin,cl)
        elseif (ivaltype==1) then
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalflo=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        else
            stop 'type523: illegal valve type'
        endif
! **** By-pass port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==1) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalbyp=rlinport(vstembyp,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalbyp=reqpport(vstembyp,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
! **** Total water-side resistances - coil + valve, by-pass + valve
        frtotflo  = frwcoil+frvalflo
        frtotbyp  = frwbypas+frvalbyp
        if (frtotflo<frsmall .or. frtotbyp<frsmall) then
! **** Total resistance almost zero
            write(*,*)&
            'type523: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Parallel combination of total flow and by-pass resistances
        frcombin  = frtotflo*frtotbyp/(frtotflo+frtotbyp)
! **** Inlet pressure from combined resistance and primary circuit flow
! **** treat as turbulent at low flows for consistency with the way valve
! **** leakage is modeled
        dp      = dpqud(frcombin,wwprim)
        pwi     = pwo+dp
! **** Flow rate through coil
        ww      = wqud(frtotflo,dp)
! **** Calculate air inlet pressure from flow resistance
        pai     = pao+dpqudlin(fra,wa)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Return water temperature
        if (wwprim>1.0e-10) then
            tret    = (ww*two + (wwprim-ww)*twi)/wwprim
        else
            tret    = two
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = pwi
        yout(7)  = ww
        yout(8)  = tret
        yout(9)  = qa
        yout(10) = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(11)  = -99.9
        else
            yout(11)  = effect
        endif
        yout(12) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(13) = ho
            yout(14) = rho/100.
            yout(15) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(13) = -99.9
            yout(14) = -99.9
            yout(15) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type523
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and three port valve
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil with a three port valve.
! *              Calculates the flow through the coil in the
! *              primary circuit from the inlet and outlet pressures.
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. pwi     : inlet water pressure                             (kpa)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. wwprim  : primary circuit water mass flow rate            (kg/s)
! *  7. ww      : water mass flow rate though coil                (kg/s)
! *  8. tret    : mixed return water temperature                     (C)
! *  9. qa      : total heat transfer to the air                    (kW)
! * 10. shr     : sensible heat ratio                                (-)
! * 11. effect  : coil effectiveness                                 (-)
! * 12. bf      : coil by-pass factor                                (-)
! * 13. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 14. rho     : outlet air relative humidity                       (-)
! * 15. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : height of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. frwbypas: by-pass water flow resistance             (0.001 kg.m)
! * 19. ivaltype: valve type: 0=lin/lin, 1=eq%(flow)/lin(byp), 2=lin/eq%
! * 20. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 21. eqpchar : valve curvature parameter (equal percentage port)  (-)
! * 22. sv      : valve rangability-ratio of highest to lowest 
! *               controllable flow (SV=1/CL, when first segment 
!                 is deadband)                                       (-)
! * 23. cl      : valve leakage (Flow fraction at second segment
! *               start position)                                    (-)
! * Shun on
! * 24. XLIN1   : stem position at which second segment starts       (-)
! * 25. XLIN2   : Third segment start position (No wearoff -- 1)     (-)
! * 26. FRAC2   : Flow fraction at third segment start position      (-)
! * 27. KVBY    : Bypass valve capacity (Kv)             (m3/hr @ 1 bar)
! * 28. CLBY    : Bypass valve leakage (Flow fraction 
!                 at second segment start position)                  (-)
! * 29. FRAC2BY : Bypass Flow fraction at third segment
!                 start position                                     (-)
! * Shun off
! * 30. coil    : 1-heating coil, 2- cooling coil                    (-)                
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT,RLINPORT3
!
!   REVISION HISTORY:    Jin Wen & Shun Li
!   DATE:                Octomber 1, 2006
!                        Shokouh Pourarian & Jin Wen
!                        April 17, 2014
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 Oct 1, 2013 Shokouh Pourarian, Drexel Uni
!
! **********************************************************************

        subroutine type533(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=15,np=30,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv,kvby,clby,frac2,frac2by,xlin1,xlin2,&
		                xlin1by,xlin2by,svby 
        integer      :: matube,mafin,coil
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,ww1,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,rlinport533,reqpport,dpqudlin,frwbypas,&
                        vstembyp,frvalflo,frvalbyp,frtotflo,frtotbyp ,&
                        dp,wqud,wwbypas,wwprim,frprim,frtotcoi,frcoibyp ,&
                        frtot,dpprim,dpqud,dpcoibyp,tret
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir,ivaltype

! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        pwi     = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
        tsdyn   = xin(9)
!        if (tsdyn>=80)then
!		    tsdyn=40
!        endif
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
        frwbypas= par_v(18)
! **** Valve characteristics
        ivaltype= par_v(19)
        kv      = par_v(20)
        eqpchar = par_v(21)
        sv      = par_v(22)
        cl      = par_v(23)
! ****  Shun on
        xlin1   = par_v(24)
        xlin2   = par_v(25)
        frac2   = par_v(26)
        kvby    = par_v(27)
        clby    = par_v(28)
        frac2by = par_v(29)
		coil    = nint(par_v(30))
!		print*,'coil', coil
!		print*,'wa', wa
!		print*,'gi', gi
! ---- Fault flag on
       if (coil==1) then
	      if (coilfault(2)==1) then
			  vstem = vcoilfault(1)
!	   WRITE(*,*) 'Heating valve is stuck'
		  endif
	   elseif (coil==2) then
	      if (coilfault(1)==1) then
			  vstem = vcoilfault(1)
		  endif
!	   WRITE(*,*) 'Cooling valve is stuck'
		endif

        if (coil==1) then
	       if (coilfault(4)==2) then
		       frwcoil = par_v(17)* vcoilfault(2)
			   frwbypas= par_v(18)* vcoilfault(2)
              
!	    WRITE(*,*) 'Heating coil water side fouling let water resistance increased'
		   endif
		elseif (coil==2) then
	       if (coilfault(3)==2) then
			   frwcoil = par_v(17)* vcoilfault(2)
			   frwbypas= par_v(18)* vcoilfault(2)
		   endif
!	    WRITE(*,*) 'Cooling coil water side fouling let water resistance increased'
		endif		
! ---- Fault flag off
! ****  Shun off
! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
! ---- Fault flag on
        if (coil==1) then
	       if (coilfault(4)==1) then
		       confin  = confin * vcoilfault(3)
			   contube = contube * vcoilfault(3)
!	    WRITE(*,*) 'Heating coil air side fouling let heat transfer rate decreased'
		   endif
		elseif (coil==2) then
	       if (coilfault(3)==1) then
			   confin  = confin * vcoilfault(3)
			   contube = contube * vcoilfault(3)
		   endif
!	    WRITE(*,*) 'Cooling coil air side fouling let heat transfer rate decreased'
		endif
		if (coil==1) then
	       if (coilfault(4)==2) then
		       confin  = confin * vcoilfault(3)
			   contube = contube * vcoilfault(3)
!	    WRITE(*,*) 'Heating coil water side fouling let heat transfer rate decreased'
		   endif
		elseif (coil==2) then
	       if (coilfault(3)==2) then
			   confin  = confin * vcoilfault(3)
			   contube = contube * vcoilfault(3)
		   endif
!	    WRITE(*,*) 'Cooling coil water side fouling let heat transfer rate decreased'
		endif			 
! ---- Fault flag off
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve types and resistances
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
        vstembyp= 1.-vstem
! **** Water flow rates through coil and bypass
        if (ivaltype>=0 .and. ivaltype<=2) then
! **** Common port carries common flow
! **** flow port - select valve type and calculate resistance
            if (ivaltype==0 .or. ivaltype==2) then
! **** Linear characteristic - two segments (linear and close-off)
               FRVALFLO=RLINPORT533(VSTEM,KV,SV,XLIN1,XLIN2,CL,FRAC2)
            elseif (ivaltype==1) then
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
                frvalflo=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
            endif
! **** By-pass port - select valve type and calculate resistance
            if (ivaltype==0 .or. ivaltype==1) then
! **** Linear characteristic - two segments (linear and close-off)
                SVBY=0.001
                XLIN1BY=1-XLIN2
                XLIN2BY=1-XLIN1
                FRVALBYP=RLINPORT533(VSTEMBYP,KVBY,SVBY,XLIN1BY,XLIN2BY,CLBY,FRAC2BY)
            else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
                frvalbyp=reqpport(vstembyp,kv,xeqp,eqpchar,xlin,sv,cl)
            endif
! **** Total water-side resistances - coil + valve, by-pass + valve
            frtotflo  = frwcoil+frvalflo
            frtotbyp  = frwbypas+frvalbyp
            if (frtotflo<frsmall .or. frtotbyp<frsmall) then
! **** Total resistance almost zero
                write(*,*)&
                'type533: water-side flow resistance must not be < ',&
                frsmall
                stop
            endif
! **** Pressure drop
            dp      = pwi-pwo
! **** Flow rate through coil
            ww1     = wqud(frtotflo,dp)
! **** Flow rate through by-pass
            wwbypas = wqud(frtotbyp,dp)
! **** Flow rate in primary circuit
            wwprim  = ww1+wwbypas
        elseif (ivaltype==3 .or. ivaltype==4) then
! **** Faulty installation
            if (ivaltype==3) then
! **** Flow and common ports swopped
! **** Resistance of flow port - equal percentage characteristic
                frprim   = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
! **** Resistance of bypass port - linear characteristic
                frvalbyp = rlinport(vstembyp,kv,sv,xlin,cl)
! **** Resistance of coil branch (common port has zero resistance)
                frtotcoi = frwcoil
! **** Resistance of bypass branch (balancing valve + bypass port)
                frtotbyp = frwbypas + frvalbyp
! **** Parallel resistance of coil and bypass branches
                frcoibyp = (frtotcoi**(-0.5)+frtotbyp**(-0.5))**2
! **** Total resistance of subsystem
                frtot    = frcoibyp + frprim
! **** Pressure drop across subsystem
                dp       = pwi-pwo
! **** Flow rate in primary circuit
                wwprim   = wqud(frtot,dp)
                dpprim   = dpqud(frprim,wwprim)
                dpcoibyp = dp - dpprim
! **** Flow rate through coil
                ww1      = wqud(frtotcoi,dpcoibyp)
! **** Flow rate through by-pass
                wwbypas  = wqud(frtotbyp,dpcoibyp)
            elseif (ivaltype==4) then
! **** Bypass and common ports swopped
! **** Resistance of flow port - equal percentage characteristic
                frvalflo = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
! **** Resistance of bypass port - linear characteristic
                frprim   = rlinport(vstembyp,kv,sv,xlin,cl)
! **** Resistance of coil branch (flow port + coil)
                frtotcoi = frvalflo + frwcoil
! **** Resistance of bypass branch (common port has zero resistance)
                frtotbyp = frwbypas
! **** Parallel resistance of coil and bypass branches
                frcoibyp = (frtotcoi**(-0.5)+frtotbyp**(-0.5))**2
! **** Total resistance of subsystem
                frtot    = frcoibyp + frprim
! **** Pressure drop across subsystem
                dp       = pwi-pwo
! **** Flow rate in primary circuit
                wwprim   = wqud(frtot,dp)
                dpprim   = dpqud(frprim,wwprim)
                dpcoibyp = dp - dpprim
! **** Flow rate through coil
                ww1      = wqud(frtotcoi,dpcoibyp)
! **** Flow rate through by-pass
                wwbypas  = wqud(frtotbyp,dpcoibyp)
            endif
        else
            stop 'valve type out of range'
        endif
! ---- Fault flag on
        if (coil==1) then
	       if (coilfault(2)==0) then
		       ww = ww1
!	    WRITE(*,*) 'Heating coil valve is fault free'
           elseif (coilfault(2)==1) then
		       ww = ww1
!	    WRITE(*,*) 'Heating coil valve stuck'
           elseif (coilfault(2)==2) then
		          if (vstem==0) then
		              ww = vcoilfault(1)
				  else
				      ww = ww1
				  endif
!	    WRITE(*,*) 'Heating coil valve leakage'
		   endif
		elseif (coil==2) then
	       if (coilfault(1)==0) then
		       ww = ww1
!	    WRITE(*,*) 'Cooling coil valve is fault free'
           elseif (coilfault(1)==1) then
		       ww = ww1
!	    WRITE(*,*) 'Cooling coil valve stuck'
           elseif (coilfault(1)==2) then
		          if (vstem==0) then
		              ww = vcoilfault(1)
				  else
				      ww = ww1
				  endif
!	    WRITE(*,*) 'Cooling coil valve leakage'
		   endif
        endif
! ---- Fault flag off

! **** Calculate air inlet pressure from flow resistance
        pai     = pao+dpqudlin(fra,wa)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Return water temperature
        if (wwprim>1.0e-10) then
            tret    = (ww*two + (wwprim-ww)*twi)/wwprim
        else
            tret    = two
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = wwprim
        yout(7)  = ww
        yout(8)  = tret
        yout(9)  = qa
        yout(10) = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(11)  = -99.9
        else
            yout(11)  = effect
        endif
        yout(12) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(13) = ho
            yout(14) = rho/100.
            yout(15) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(13) = -99.9
            yout(14) = -99.9
            yout(15) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type533
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and three port valve
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil with a three port valve.
! *              Calculates inlet pressure and coil water flow from
! *              primary circuit flow rate and outlet pressure
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. wwprim  : primary circuit water mass flow rate            (kg/s)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. ww      : water mass flow rate though coil                (kg/s)
! *  8. tret    : mixed return water temperature                     (C)
! *  9. qa      : total heat transfer to the air                    (kW)
! * 10. shr     : sensible heat ratio                                (-)
! * 11. effect  : coil effectiveness                                 (-)
! * 12. bf      : coil by-pass factor                                (-)
! * 13. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 14. rho     : outlet air relative humidity                       (-)
! * 15. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : height of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. frwbypas: by-pass water flow resistance             (0.001 kg.m)
! * 19. ivaltype: valve type: 0=lin/lin, 1=eq%(flow)/lin(byp), 2=lin/eq%
! * 20. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 21. eqpchar : valve curvature parameter (equal percentage port)  (-)
! * 22. sv      : valve rangability                                  (-)
! * 23. cl      : valve leakage (fractional flow)                    (-)
! * 24. mode    : 1= cooling coil, 2= heating coil                   (-)
!
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type529(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=15,np=24,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,ww1,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,wwprim,frwbypas,&
                        vstembyp,frvalflo,frvalbyp,frtotflo,frtotbyp,&
                        frcombin,dp,dpqud,wqud,tret
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir,ivaltype,mode

! **** Read in parameters
! **** It is serving as cooling coil or heating coil
        mode    = par_v(24)
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
! *** Fault flag on*********************************
	    if (mode.EQ.1) then
		    if (Coolingfoul.EQ.2) then
! *** Cooling coil water side fouling let water resistance increased
			frwcoil = Vcoolingfoul1*par_v(17)
			endif
		elseif (mode.EQ.2) then
		    if (Heatingfoul.EQ.2) then
! *** Heating coil water side fouling let water resistance increased
			frwcoil = Vheatingfoul1*par_v(17)
			endif
		endif
! *** Fault flag off*********************************	
        frwbypas= par_v(18)
! **** Valve characteristics
        ivaltype= par_v(19)
        kv      = par_v(20)
        eqpchar = par_v(21)
        sv      = par_v(22)
        cl      = par_v(23)
		
! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        wwprim  = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
! *** Fault flag on*********************************
		if (mode.EQ.1) then
		    if (VLVcoolcoil.EQ.1) then
! *** Cooling valve is stuck
			vstem = VVLVcoolcoil
			endif
		elseif (mode.EQ.2) then
		    if (VLVheatcoil.EQ.1) then
! *** Heating valve is stuck
			vstem = VVLVheatcoil
			endif
		endif
! *** Fault flag off*********************************
        tsdyn   = xin(9)

! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
! *** Fault flag on*********************************
		if (mode.EQ.1) then
		    if (Coolingfoul.EQ.1) then
! *** Cooling coil air side fouling let heat transfer rate decreased
			    confin  = Vcoolingfoul2*confin
			    contube = Vcoolingfoul2*contube

	        elseif (Coolingfoul.EQ.2) then
! *** Cooling coil water side fouling let heat transfer rate decreased
			    confin  = Vcoolingfoul2*confin
			    contube = Vcoolingfoul2*contube
		    endif
		    
		elseif (mode.EQ.2) then
		    if (Heatingfoul.EQ.1) then
! *** heating coil air side fouling let heat transfer rate decreased
			    confin  = Vheatingfoul2*confin
			    contube = Vheatingfoul2*contube

	        elseif (Heatingfoul.EQ.2) then
			    confin  = Vheatingfoul2*confin
			    contube = Vheatingfoul2*contube
		    endif
		endif
! *** Fault flag off*********************************
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve types and resistances
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
        vstembyp= 1.-vstem
! **** Flow port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==2) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalflo=rlinport(vstem,kv,sv,xlin,cl)
        elseif (ivaltype==1) then
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalflo=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
        else
            stop 'type520: illegal valve type'
        endif
! **** By-pass port - select valve type and calculate resistance
        if (ivaltype==0 .or. ivaltype==1) then
! **** Linear characteristic - two segments (linear and close-off)
            frvalbyp=rlinport(vstembyp,kv,sv,xlin,cl)
        else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
            frvalbyp=reqpport(vstembyp,kv,xeqp,eqpchar,xlin,sv,cl)
        endif
! **** Total water-side resistances - coil + valve, by-pass + valve
        frtotflo  = frwcoil+frvalflo
        frtotbyp  = frwbypas+frvalbyp
        if (frtotflo<frsmall .or. frtotbyp<frsmall) then
! **** Total resistance almost zero
            write(*,*)&
            'type520: water-side flow resistance must not be < ',frsmall
            stop
        endif
! **** Parallel combination of total flow and by-pass resistances
        frcombin  = frtotflo*frtotbyp/(frtotflo+frtotbyp)
! **** Inlet pressure from combined resistance and primary circuit flow
! **** treat as turbulent at low flows for consistency with the way valve
! **** leakage is modeled
        dp      = dpqud(frcombin,wwprim)
        pwi     = pwo+dp
! **** Flow rate through coil
        ww1     = wqud(frtotflo,dp)

! **** Fault flag on****************************************
        if (mode.EQ.1) then
		    if (VLVcoolcoil.EQ.0) then
! *** Cooling valve is fault free
			    ww = ww1
            elseif (VLVcoolcoil.EQ.1) then
! *** Cooling valve is stuck
			    ww = ww1
            elseif (VLVcoolcoil.EQ.2) then
! *** Cooling valve is leaking
                if (vstem.EQ.0) then
                      ww = VVLVcoolcoil
                else
				      ww = ww1
                endif
			endif
		elseif (mode.EQ.2) then
		    if (VLVheatcoil.EQ.0) then
! *** Heating valve is fault free
			    ww = ww1
            elseif (VLVheatcoil.EQ.1) then
! *** Heating valve is stuck
			    ww = ww1
            elseif (VLVheatcoil.EQ.2) then
! *** Heating valve is leaking
                if (vstem.EQ.0) then
                      ww = VVLVheatcoil
                else
				      ww = ww1
                endif
			endif
        endif
! **** Fault flag off****************************************
! **** Calculate air inlet pressure from flow resistance
        pai     = pao+dpqudlin(fra,wa)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Return water temperature
        if (wwprim>1.0e-10) then
            tret    = (ww*two + (wwprim-ww)*twi)/wwprim
        else
            tret    = two
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = pwi
        yout(7)  = ww
        yout(8)  = tret
        yout(9)  = qa
        yout(10) = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(11)  = -99.9
        else
            yout(11)  = effect
        endif
        yout(12) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(13) = ho
            yout(14) = rho/100.
            yout(15) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(13) = -99.9
            yout(14) = -99.9
            yout(15) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type529
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Dynamic or steady state coil and three port valve
! *
! * PURPOSE:     Calculate the outlet air and water conditions for a
! *              heating or cooling coil with a three port valve.
! *              Calculates the flow through the coil and in the
! *              primary circuit from the inlet and outlet pressures.
! **********************************************************************
! * INPUTS
! * ======
! *  1. tai     : inlet air dry bulb temperature                     (C)
! *  2. gi      : inlet air humidity ratio                       (kg/kg)
! *  3. pao     : outlet air pressure                              (kPa)
! *  4. wa      : dry air mass flow rate                          (kg/s)
! *  5. twi     : inlet water temperature                            (C)
! *  6. pwi     : inlet water pressure                             (kPa)
! *  7. pwo     : outlet water pressure                            (kPa)
! *  8. vstem   : valve stem position                                (-)
! *  9. tsdyn   : effective coil surface temperature                 (C)
! *
! * OUTPUTS
! * =======
! *  1. ts      : effective coil surface temperature                 (C)
! *  2. tao     : outlet air dry bulb temperature                    (C)
! *  3. go      : outlet air humidity ratio                      (kg/kg)
! *  4. pai     : inlet air pressure                               (kPa)
! *  5. two     : outlet water temperature                           (C)
! *  6. wwprim  : primary circuit water mass flow rate            (kg/s)
! *  7. ww      : water mass flow rate though coil                (kg/s)
! *  8. tret    : mixed return water temperature                     (C)
! *  9. qa      : total heat transfer to the air                    (kW)
! * 10. shr     : sensible heat ratio                                (-)
! * 11. effect  : coil effectiveness                                 (-)
! * 12. bf      : coil by-pass factor                                (-)
! * 13. ho      : outlet air specific enthalpy                   (kJ/kg)
! * 14. rho     : outlet air relative humidity                       (-)
! * 15. twbo    : outlet air wet-bulb temperature                    (C)
! *
! * PARAMETERS
! * ==========
! *  1. dynamic : 0 = steady state, 1 = dynamic                      (-)
! *  2. ifault  : 0 = no faults, 1 = parallel flow (cooling coils)   (-)
! *  3. psychro : FALSE = no psychrometric output calcs, TRUE = calcs(-)
! *  4. nrow    : number of rows of tubes                            (-)
! *  5. ntpr    : number of tubes per row                            (-)
! *  6. ncir    : number of parallel water circuits                  (-)
! *  7. lcoil   : length of finned section in direction of flow      (m)
! *  8. hcoil   : height of finned section                           (m)
! *  9. wcoil   : width of finned section                            (m)
! * 10. dotube  : tube outside diameter                              (m)
! * 11. thitube : tube wall thickness                                (m)
! * 12. watube  : tube material (Al=1,Cu=2,Fe=3,CaCO3=4)             (-)
! * 13. spafin  : fin spacing (pitch)                                (m)
! * 14. thifin  : fin thickness                                      (m)
! * 15. wafin   : fin material (Al=1,Cu=2,Fe=3)                      (-)
! * 16. fra     : flow resistance on air side               (0.001 kg.m)
! * 17. frwcoil : coil water flow resistance                (0.001 kg.m)
! * 18. frwbypas: by-pass water flow resistance             (0.001 kg.m)
! * 19. ivaltype: valve type: 0=lin/lin, 1=eq%(flow)/lin(byp), 2=lin/eq%
! * 20. kv      : valve capacity (Kv)                    (m3/hr @ 1 bar)
! * 21. eqpchar : valve curvature parameter (equal percentage port)  (-)
! * 22. sv      : valve rangability                                  (-)
! * 23. cl      : valve leakage (fractional flow)                    (-)
! *
! * SAVED (dynamic mode only)
! * =====
! *  1. time    : time of previous call
! *  2. dt      : solution of differential equation from previous call
! *  3. dtp     : solution of DE from previous step time
! *  4. aa      : A coefficent in dT/dt=A*T+B from previous call
! *  5. aap     : A coefficent in dT/dt=A*T+B from previous step time
! *  6. bb      : B coefficent in dT/dt=A*T+B from previous call
! *  7. bbp     : B coefficent in dT/dt=A*T+B from previous step time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  Assumes coil is all dry or all wet
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 14, 1995
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  LICOILDY,LICOILSS
!   FUNCTIONS  CALLED:   RLINPORT,REQPPORT
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Cooling Coil Models to be used in Transient
!                        and/or Wet Regimes -- Theorical Analysis and
!                        Experimental Validation.
!                        X.DING, J-P EPPE,J.LEBRUN,M.WASACZ.
!                        I.E.A. ANNEX 17 document AN17-901019-01.
!                        University of Liege, Belgium.
!
!                        (adapted from the cooling coil model written
!                        by V. Corrado - Polytechnic of Turin, Italy)
!
! **********************************************************************
! * INTERNAL VARIABLES
! * ==================
! *
! * Material Properties
! * -------------------
! * volcpm  : volumic specific heat of Al, Cu and mild steel     (kJ/m3)
! * conma   : thermal conductivity of Al, Cu and mild steel   (kW/(m.K))
! *
! * Geometrical characteristics
! * ---------------------------
! * fracfin : fraction of face area blocked by fins                  (-)
! * acgross : gross face area                                       (m2)
! * ntube   : total number of tubes                                  (-)
! * ttubelen: total tube length                                      (m)
! * axo     : gross outside tube surface area                       (m2)
! * ap      : net outside tube surface area, allowing for fins      (m2)
! * as      : net air side fin area                                 (m2)
! * ditube  : tube inside diameter                                   (m)
! * ai      : inside surface area of tubes                          (m2)
! * afree   : exchanger minimum free-flow area on air side          (m2)
! * aext    : exchanger total heat transfer area on air side        (m2)
! * aratio  : ratio of total fin area to total heat transfer area    (-)
! * bratio  : ratio of air side total area : water side internal area(-)
! * rifin   : fin inside radius                                      (m)
! * rofin   : effective fin radius                                   (m)
! * heifin  : effective fin height                                   (m)
! * aflow   : flow area on water side                               (m2)
! * hydiam  : coil hydraulic diameter                                (m)
! * confin  : thermal conductivity of fin material            (kW/(m.K))
! * contube : thermal conductivity of tube material           (kW/(m.K))
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type524(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=1,&
                                             ni=9,no=15,np=23,&
                                             ns=1+ndiffeq*6
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: lcoil,kv
        integer      :: matube,mafin
        logical      :: dynamic,psychro
! **** Thermophysical constants
        real,dimension(4)   :: volcpm=(/2455.,3475.,3915.,2250./),&
                               conma=(/0.221,0.393,0.0453,0.0013/)
! **** Miscellaneous constants (pi, atmospheric pressure, minimum
! **** water-side resistance
        real         :: pi=3.14159,patmpa=101325.,frsmall=1.0e-8
! **** Break-points in valve characteristics (see documentation for
! **** type329 two port valve)
        real         :: xlin=0.01,xeqp=0.33
        real         :: eqpchar,sv,cl,fracfin,acgross,ttubelen,axo,asx,&
                        ap,ditube,ai,afree,aext,aratio,bratio,rifin,&
                        rofin,heifin,aflow,hydiam,confin,contube,vcpfin,&
                        vcptube,frwvalve,frwtot,pai,pwi,tsp,aap,bbp,&
                        twbo,rho,ho,bf,shr,qa,two,go,tao,bb,aa,aabar,&
                        bbbar,tsbar,ts,effect,tai,gi,gsat,fwphi,pao,&
                        wa,twi,ww,pwo,vstem,tsdyn,hcoil,wcoil,&
                        dotube,thitube,spafin,thifin,fra,frwcoil,&
                        rlinport,reqpport,dpqudlin,frwbypas,&
                        vstembyp,frvalflo,frvalbyp,frtotflo,frtotbyp ,&
                        dp,wqud,wwbypas,wwprim,frprim,frtotcoi,frcoibyp ,&
                        frtot,dpprim,dpqud,dpcoibyp,tret
        integer      :: i,icountfl,ntube,is,ifault,nrow,ntpr,ncir,ivaltype

! **** Read in inputs
        tai     = xin(1)
        gi      = xin(2)
! **** Limit humidity ratio to saturated value
        gsat    = fwphi(tai,100.0,patmpa)
        gi      = min(gi,gsat)
        pao     = xin(3)
        wa      = xin(4)
        twi     = xin(5)
        pwi     = xin(6)
        pwo     = xin(7)
        vstem   = xin(8)
        tsdyn   = xin(9)
! **** Read in parameters
! **** Dynamic or steady state
        if (nint(par_v(1))==1) then
            dynamic = .true.
        else
            dynamic = .false.
        endif
! **** Fault type
        ifault = nint(par_v(2))
! **** Peform additional pyschrometric calculations?
        if (nint(par_v(3))/=0) then
            psychro = .true.
        else
            psychro = .false.
        endif
! **** Coil geometry and materials
        nrow    = nint(par_v(4))
        ntpr    = nint(par_v(5))
        ncir    = nint(par_v(6))
        lcoil   = par_v(7)
        hcoil   = par_v(8)
        wcoil   = par_v(9)
        dotube  = par_v(10)
        thitube = par_v(11)
        matube  = nint(par_v(12))
        spafin  = par_v(13)
        thifin  = par_v(14)
        mafin   = nint(par_v(15))
! **** Air and water flow resistances
        fra     = par_v(16)
        frwcoil = par_v(17)
        frwbypas= par_v(18)
! **** Valve characteristics
        ivaltype= par_v(19)
        kv      = par_v(20)
        eqpchar = par_v(21)
        sv      = par_v(22)
        cl      = par_v(23)
! **** Heat exchanger geometry
        if (nrow<=2) then
! **** One or two rows -> cross flow
            icountfl=0
        elseif (nrow>2) then
            if (ifault/=1) then
! **** Three or more rows and no connection fault -> counterflow
                icountfl=1
            else
! **** Three or more rows and connection fault -> parallel flow
                icountfl=-1
            endif
        endif
! **** Heat transfer areas
        fracfin = thifin/spafin
        acgross = hcoil*wcoil
        ntube   = nrow*ntpr
        ttubelen= ntube*wcoil
        axo     = pi*dotube*dotube/4.0
        ap      = pi*dotube*ttubelen*(1.0-fracfin)
        asx     = 2.0*(hcoil*lcoil-ntube*axo)*wcoil/spafin
        ditube  = dotube-2.0*thitube
        ai      = pi*ditube*ttubelen
        afree   = acgross*(1.0-ntpr*dotube/hcoil)*(1.0-fracfin)
        aext    = asx+ap
        aratio  = asx/aext
        bratio  = aext/ai
        rifin   = dotube/2.0
        rofin   = sqrt(hcoil*lcoil/(pi*ntube))
        heifin  = rofin-rifin
        aflow   = ncir*pi*ditube*ditube/4.0
        hydiam  = 4.0*afree*lcoil/aext
        confin  = conma(mafin)
        contube = conma(matube)
        vcpfin  = volcpm(mafin)
        vcptube = volcpm(matube)
! **** Valve types and resistances
! **** Limit stem position to range 0-1
        vstem   = max( 0.0, min(1.0,vstem) )
        vstembyp= 1.-vstem
! **** Water flow rates through coil and bypass
        if (ivaltype>=0 .and. ivaltype<=2) then
! **** Common port carries common flow
! **** flow port - select valve type and calculate resistance
            if (ivaltype==0 .or. ivaltype==2) then
! **** Linear characteristic - two segments (linear and close-off)
                frvalflo=rlinport(vstem,kv,sv,xlin,cl)
            elseif (ivaltype==1) then
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
                frvalflo=reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
            endif
! **** By-pass port - select valve type and calculate resistance
            if (ivaltype==0 .or. ivaltype==1) then
! **** Linear characteristic - two segments (linear and close-off)
                frvalbyp=rlinport(vstembyp,kv,sv,xlin,cl)
            else
! **** Equal percentage characteristic - three segments (exponential,
! **** linear and close-off)
                frvalbyp=reqpport(vstembyp,kv,xeqp,eqpchar,xlin,sv,cl)
            endif
! **** Total water-side resistances - coil + valve, by-pass + valve
            frtotflo  = frwcoil+frvalflo
            frtotbyp  = frwbypas+frvalbyp
            if (frtotflo<frsmall .or. frtotbyp<frsmall) then
! **** Total resistance almost zero
                write(*,*)&
                'type524: water-side flow resistance must not be < ',&
                frsmall
                stop
            endif
! **** Pressure drop
            dp      = pwi-pwo
! **** Flow rate through coil
            ww      = wqud(frtotflo,dp)
! **** Flow rate through by-pass
            wwbypas = wqud(frtotbyp,dp)
! **** Flow rate in primary circuit
            wwprim  = ww+wwbypas
        elseif (ivaltype==3 .or. ivaltype==4) then
! **** Faulty installation
            if (ivaltype==3) then
! **** Flow and common ports swopped
! **** Resistance of flow port - equal percentage characteristic
                frprim   = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
! **** Resistance of bypass port - linear characteristic
                frvalbyp = rlinport(vstembyp,kv,sv,xlin,cl)
! **** Resistance of coil branch (common port has zero resistance)
                frtotcoi = frwcoil
! **** Resistance of bypass branch (balancing valve + bypass port)
                frtotbyp = frwbypas + frvalbyp
! **** Parallel resistance of coil and bypass branches
                frcoibyp = (frtotcoi**(-0.5)+frtotbyp**(-0.5))**2
! **** Total resistance of subsystem
                frtot    = frcoibyp + frprim
! **** Pressure drop across subsystem
                dp       = pwi-pwo
! **** Flow rate in primary circuit
                wwprim   = wqud(frtot,dp)
                dpprim   = dpqud(frprim,wwprim)
                dpcoibyp = dp - dpprim
! **** Flow rate through coil
                ww       = wqud(frtotcoi,dpcoibyp)
! **** Flow rate through by-pass
                wwbypas  = wqud(frtotbyp,dpcoibyp)
            elseif (ivaltype==4) then
! **** Bypass and common ports swopped
! **** Resistance of flow port - equal percentage characteristic
                frvalflo = reqpport(vstem,kv,xeqp,eqpchar,xlin,sv,cl)
! **** Resistance of bypass port - linear characteristic
                frprim   = rlinport(vstembyp,kv,sv,xlin,cl)
! **** Resistance of coil branch (flow port + coil)
                frtotcoi = frvalflo + frwcoil
! **** Resistance of bypass branch (common port has zero resistance)
                frtotbyp = frwbypas
! **** Parallel resistance of coil and bypass branches
                frcoibyp = (frtotcoi**(-0.5)+frtotbyp**(-0.5))**2
! **** Total resistance of subsystem
                frtot    = frcoibyp + frprim
! **** Pressure drop across subsystem
                dp       = pwi-pwo
! **** Flow rate in primary circuit
                wwprim   = wqud(frtot,dp)
                dpprim   = dpqud(frprim,wwprim)
                dpcoibyp = dp - dpprim
! **** Flow rate through coil
                ww       = wqud(frtotcoi,dpcoibyp)
! **** Flow rate through by-pass
                wwbypas  = wqud(frtotbyp,dpcoibyp)
            endif
        else
            stop 'valve type out of range'
        endif
! **** Calculate air inlet pressure from flow resistance
        pai     = pao+dpqudlin(fra,wa)
! **** Calculate coil duty and outlet conditions
        if (dynamic) then
! **** Initialize at beginning of simulation
            if (itime<=1) then
                if (init==0 .or. saved_v(1)>time) then
                    saved_v(1) = -999999.
               endif
                if (init==0) then
! **** Capacitance of coil initially at inlet air temperature (coil "off")
                    saved_v(2) = tai
                    saved_v(4) = 0.0
                    saved_v(6) = 0.0
                endif
            endif
! **** Every time-step
            if (time>saved_v(1)) then
! **** First call of time-step - update value of temperature rise, aa and bb
! **** from previous time-step
                do is = 2,ns-1,2
                    saved_v(is+1) = saved_v(is)
                enddo
            endif
! **** Update previous values
            tsp = saved_v(3)
            aap = saved_v(5)
            bbp = saved_v(7)
            call licoilab(tai,gi,wa,twi,ww,tsdyn,&
                          psychro,icountfl,lcoil,wcoil,dotube,thitube,&
                          thifin,ttubelen,asx,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,vcpfin,vcptube,&
                          aa,bb,tao,go,two,qa,shr,bf,ho,rho,twbo)
! **** Integrate differential equation for surface temperature
            aabar = (aa + aap)/2.0
            bbbar = (bb + bbp)/2.0
            call diffeq(time,aabar,bbbar,tsp,ts,tsbar)
! **** Save time of current call
            saved_v(1) = time
! **** Save provisional value to be used in updating at next step time
            saved_v(2) = ts
            saved_v(4) = aa
            saved_v(6) = bb
        else
! **** Steady state model - output is value of surface temperature
            call licoilss(tai,gi,wa,twi,ww,&
                          psychro,icountfl,lcoil,wcoil,thitube,&
                          thifin,ditube,afree,aext,&
                          aratio,bratio,rifin,rofin,heifin,aflow,&
                          hydiam,confin,contube,&
                          ts,tao,go,two,qa,shr,effect,bf,ho,rho,twbo)
        endif
! **** Return water temperature
        if (wwprim>1.0e-10) then
            tret    = (ww*two + (wwprim-ww)*twi)/wwprim
        else
            tret    = two
        endif
! **** Assign output values
        yout(1)  = ts
        yout(2)  = tao
        yout(3)  = go
        yout(4)  = pai
        yout(5)  = two
        yout(6)  = wwprim
        yout(7)  = ww
        yout(8)  = tret
        yout(9)  = qa
        yout(10) = shr
        if (dynamic) then
! **** Effectiveness not defined for dynamic case
            yout(11)  = -99.9
        else
            yout(11)  = effect
        endif
        yout(12) = bf
        if (psychro) then
! **** Auxilliary outputs
            yout(13) = ho
            yout(14) = rho/100.
            yout(15) = twbo
        else
! **** Auxilliary outputs not calculated - set values to flag this
            yout(13) = -99.9
            yout(14) = -99.9
            yout(15) = -99.9
        endif
! **** Allow freezing of algebraic variables
        if (.not.dynamic) then
            iostat(1)=1
        else
            iostat(1)=0
        endif
        do i=2,no
            iostat(i)=1
        enddo
        return
        end subroutine type524

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent VAV box
! *
! * PURPOSE:        Calculate the inlet pressure of a motor-driven,
! *                 pressure-independent VAV box from the outlet
! *                 pressure, the mass flow rate and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wair    : air mass flow rate                             (kg/s)
! *  2. pout    : outlet pressure                                 (kPa)
! *  3. fflowset: demanded flowrate (normalized to nominal)       (0-1)
! *
! * OUTPUTS
! * =======
! *  1. pin     : inlet pressure                                  (kPa)
! *  2. damppos : damper position (0=closed, 1=open)                (-)
! *  3. fsped   : fractional motor velocity (0-1)                   (-)
! *  4. tssrev  : number of stops/starts/reversals                  (-)
! *  5. ttrav   : total distance travelled by valve/damper          (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                   (m3/s)
! *  2. farea   : face area of damper(s)                           (m2)
! *  3. pdopen  : pressure drop at nominal flow with damper open  (kPa)
! *  4. ttran   : travel time (0-90 deg)                            (s)
! *  5. fspedmin: minimum fractional motor speed                    (-)
! *  6. hys     : hysteresis                                        (-)
! *  7. kp      : controller gain (frac speed/frac error)           (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type525(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=3,no=5,np=7,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: volumnom,farea,pdopen,ttran,fspedmin,hys,actposp,&
                        fspedp,tssrevp,ttravp,dampposp,fflow,fspeddem,&
                        fsped,fspedbar,actpos,tssrev,ttrav,damppos,ropen,&
                        alegg1,blegg1,r,radd,rtot,pin,wair,pout,fflowset,&
                        spedlim,clip,rdamper,dpqudlin
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        wair     = xin(1)
        pout     = xin(2)
        fflowset = xin(3)
! **** Read in par_vameters
        volumnom = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 525: nominal flow rate must be greater than zero'
        endif
        farea    = par_v(2)
        pdopen   = par_v(3)
        ttran    = par_v(4)
        fspedmin = par_v(5)
        hys      = par_v(6)
        kp       = par_v(7)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
        rtot = r+radd
! **** Pressure drop and inlet pressure
        pin = pout+dpqudlin(rtot,wair)
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = pin
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type525

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Pressure-independent VAV box - calculates flow
! *
! * PURPOSE:        Calculate the flow rate through a motor-driven,
! *                 pressure-independent VAV box from the inlet
! *                 pressure, the outlet pressure and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. pin     : inlet pressure                                  (kPa)
! *  2. pout    : outlet pressure                                 (kPa)
! *  3. fflowset: demanded flowrate (normalized to nominal)       (0-1)
! *  4. wair    : air mass flow rate                             (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. wair    : air mass flow rate                             (kg/s)
! *  2. damppos : damper position (0=closed, 1=open)                (-)
! *  3. fsped   : fractional motor velocity (0-1)                   (-)
! *  4. tssrev  : number of stops/starts/reversals                  (-)
! *  5. ttrav   : total distance travelled by valve/damper          (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                   (m3/s)
! *  2. farea   : face area of damper(s)                           (m2)
! *  3. pdopen  : pressure drop at nominal flow with damper open  (kPa)
! *  4. ttran   : travel time (0-90 deg)                            (s)
! *  5. fspedmin: minimum fractional motor speed                    (-)
! *  6. hys     : hysteresis                                        (-)
! *  7. kp      : controller gain (frac speed/frac error)           (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type526(xin,yout,par_v,saved_v,iostat)

         use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=4,no=5,np=7,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: pin,pout,fflowset,wair,volumnom,farea,pdopen,&
                        ttran,fspedmin,hys,actposp,fspedp,&
                        tssrevp,ttravp,dampposp,fflow,fspeddem,fsped,&
                        spedlim,fspedbar,actpos,clip,tssrev,ttrav,&
                        damppos,ropen,alegg1,blegg1,r,rdamper,radd,&
                        rtot,dp,wqudlin
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        pin      = xin(1)
        pout     = xin(2)
        fflowset = xin(3)
        wair     = xin(4)
! **** Read in parameters
        volumnom = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 526: nominal flow rate must be greater than zero'
        endif
        farea    = par_v(2)
        pdopen   = par_v(3)
        ttran    = par_v(4)
        fspedmin = par_v(5)
        hys      = par_v(6)
        kp       = par_v(7)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
!       fspedbar = fspedp
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
        rtot = r+radd
! **** Pressure drop and mass flow rate
        dp   = pin-pout
        wair = wqudlin(rtot,dp)
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = wair
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type526
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Pressure-independent VAV box (implicit flow)
! *
! * PURPOSE:        Calculate the flow rate through a motor-driven,
! *                 pressure-independent VAV box from the design
! *                 pressure drop and the demanded normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. fflowset: demanded flowrate (normalized to nominal)       (0-1)
! *  2. wair    : air mass flow rate                             (kg/s)
! *  3. sfstatus: supply fan status (1 = on, 0 = off)               (-)
! *
! * OUTPUTS
! * =======
! *  1. wair    : mass flow rate                                 (kg/s)
! *  2. damppos : damper position (0=closed, 1=open)                (-)
! *  3. fsped   : fractional motor velocity (0-1)                   (-)
! *  4. tssrev  : number of stops/starts/reversals                  (-)
! *  5. ttrav   : total distance travelled by valve/damper          (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                   (m3/s)
! *  2. farea   : face area of damper(s)                           (m2)
! *  3. pdopen  : pressure drop at nominal flow with damper open  (kPa)
! *  4. ttran   : travel time (0-90 deg)                            (s)
! *  5. fspedmin: minimum fractional motor speed                    (-)
! *  6. hys     : hysteresis                                        (-)
! *  7. kp      : controller gain (frac speed/frac error)           (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
!**********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type527(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=3,no=5,np=7,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kp
        logical      :: con,sfstatus,retolog
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: fflowset,wair,volumnom,farea,pdopen,ttran,&
                        fspedmin,hys,actposp,fspedp,tssrevp,ttravp,&
                        dampposp,fflow,fspeddem,fsped,spedlim,fspedbar,&
                        actpos,clip,tssrev,ttrav,damppos,ropen,alegg1,&
                        blegg1,r,rdamper,radd,rtot,wqudlin
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        fflowset = xin(1)
        wair     = xin(2)
        sfstatus = retolog(xin(3))
! **** Read in parameters
        volumnom = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 527: nominal flow rate must be greater than zero'
        endif
        farea    = par_v(2)
        pdopen   = par_v(3)
        ttran    = par_v(4)
        fspedmin = par_v(5)
        hys      = par_v(6)
        kp       = par_v(7)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1)  = 0.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  Hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Calculate damper resistance and flow rate if supply fan on
        if (sfstatus) then
! **** Resistance when fully open
            ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
            alegg1 = alegg(0)            ! added 12/23/99
            blegg1 = blegg(0)            ! added 12/23/99
            r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
            radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
            rtot = r+radd
! **** Calculate flow rate from total resistance and external pressure drop
           wair = wqudlin(rtot,pdopen)
        else
! **** Supply fan off - set flow rate to zero
            wair = 0.0
        endif
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = wair
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow not changing
        con = (iostat(1)<-1).or.(iostat(1)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type527

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Flow split and pressure-independent VAV box
! *
! * PURPOSE:        Calculate the outlet flow rates and the inlet pressure
! *                 for a flow split with a motorized pressure-independent
! *                 VAV box in one branch, given the outlet pressures, the
! *                 inlet flow rate and the demanded normalized flow rate
! *                 through the VAV box.
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. wi      : inlet mass flow rate                            (kg/s)
! *  2. pomain  : pressure at main outlet                          (kPa)
! *  3. povav   : pressure at VAV box outlet                       (kPa)
! *  4. fflowset: demanded VAV flowrate (normalized to nominal)    (0-1)
! *  5. wovav   : mass flow rate through VAV box                  (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. pi      : inlet pressure                                   (kPa)
! *  2. womain  : mass flow rate at main outlet                   (kg/s)
! *  3. wovav   : mass flow rate through VAV box                  (kg/s)
! *  4. damppos : damper position (0=closed, 1=open)                 (-)
! *  5. fsped   : fractional motor velocity (0-1)                    (-)
! *  6. tssrev  : number of stops/starts/reversals                   (-)
! *  7. ttrav   : total distance travelled by valve/damper           (-)
! *
! * PARAMETERS
! * ==========
! *  1. rin     : inlet resistance                           (0.001/k.m)
! *  2. rmain   : resistance of main outlet                  (0.001/k.m)
! *  3. rbranch : resistance of branch to VAV box            (0.001/k.m)
! *  4. volumnom: nominal volumetric flow rate for VAV box        (m3/s)
! *  5. farea   : face area of damper(s)                            (m2)
! *  6. pdopen  : box pressure drop at nom. flow with damper open  (kPa)
! *  7. ttran   : actuator travel time (0-90 deg)                    (s)
! *  8. fspedmin: minimum fractional motor speed                     (-)
! *  9. hys     : hysteresis                                         (-)
! * 10. kp      : controller gain (frac speed/frac error)            (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:  FLOWSPLT
!   FUNCTIONS  CALLED:   HYSTRSIS
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************
! *
! * INTERNAL VARIABLES
! * ==================
! * wcritu  : flow threshold, lower limit of pure quadratic solution
! * wcritl  : flow threshold, upper limit of pure linear solution
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type528(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=5,no=7,np=10,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: wcritl=3.e-2, wcritu=6.e-2
        real         :: wi,pomain,povav,fflowset,wovav,rin,rmain,&
                        rbranch,volumnom,farea,pdopen,ttran,fspedmin,&
                        hys,actposp,fspedp,tssrevp,ttravp,dampposp,&
                        fflow,fspeddem,fsped,spedlim,fspedbar,actpos,&
                        clip,tssrev,ttrav,damppos,ropen,alegg1,blegg1,&
                        r,rdamper,radd,rvav,womain,pi
        integer      :: i,is,ispedp,isped,ifail

! **** Read in inputs
        wi       = xin(1)
        pomain   = xin(2)
        povav    = xin(3)
        fflowset = xin(4)
        wovav    = xin(5)
! **** Check for reverse inlet flow at last iteration
        if (wi<-0.01 .and. lastit==1)&
            write(*,*) 'warning:', ' type 528: ',&
                       'reverse inlet flow'
! **** Read in parameters
        rin      = par_v(1)
        rmain    = par_v(2)
        rbranch  = par_v(3)
        volumnom = par_v(4)
        if (volumnom<=0.0) then
            stop 'type 528: nominal flow rate must be greater than zero'
        endif
        farea    = par_v(5)
        pdopen   = par_v(6)
        ttran    = par_v(7)
        fspedmin = par_v(8)
        hys      = par_v(9)
        kp       = par_v(10)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wovav/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** Control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Increment stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Increment distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  Hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)               ! added 12/23/1999
        blegg1 = blegg(0)               ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
        rvav = r+radd
! **** Call flow split routine
        call flowsplt(wi,pomain,povav,rin,rmain,rvav,wcritl,wcritu,&
                      rtolx,pi,womain,wovav,ifail)
! **** Check for unsuccesful completion
        if (ifail==1) then
! **** One or more resistances negative
            write(*,*) ' type 528: ',&
                       'resistances must not be negative'
            stop
        elseif (ifail==2) then
! **** Zero resistance for both outlet branches
            write(*,*) ' type 528: ',&
                       'both outlet resistances cannot be zero'
            stop
        endif
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = pi
        yout(2) = womain
        yout(3) = wovav
        yout(4) = damppos
        yout(5) = fsped
        yout(6) = tssrev
        yout(7) = ttrav
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(4)<-1).or.(iostat(4)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type528
!**********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! *
! * SUBROUTINE: Motorized pressure-independent dual duct box 1 
! *             (calculates flow)
! *
! * PURPOSE:     Calculate the outlet flow rates for a VAV mixing box, 
! *              given inlet and outlet pressures
! *
!***********************************************************************
! * INPUTS
! * ======
! *  1. PI1     : Inlet pressure 1                                 (kpa)
! *  2. PI2     : Inlet pressure 2                                 (kpa)
! *  3. PO      : Pressure at outlet                               (kPa)
! *  4. C1      : damper 1 position (0=closed, 1=open)               (-)
! *  5. C2      : damper 2 position (0=closed, 1=open)               (-)
! *
! * OUTPUTS
! * =======
! *  1. WI1     : Inlet mass flow rate 1                          (kg/s)
! *  2. WI2     : Inlet mass flow rate 2                          (kg/s)
! *  3. WO      : Mass flow rate at outlet                        (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. RFIX1   : Fixed resistance in branch1                (0.001/k.m)
! *  2. RFIX2   : Fixed resistance in branch2                (0.001/k.m)
! *  3. FAREA1  : Face area of damper1                              (m2)
! *  4. FAREA2  : Face area of damper2                              (m2)
!***********************************************************************
!
!   MAJOR RESTRICTIONS:  None
!
!   DEVELOPER:           XIWANG LI
!
!   DATE:                
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWMERG
!   FUNCTIONS  CALLED:   RDAMPER
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
!***********************************************************************

        subroutine type535(xin,yout,par_v,saved_v,iostat)
	    use modsim_head
        implicit none
        integer,parameter                 :: ni=5,no=3,np=4,ns=0
		real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: pi1,pi2,po,wi1,wi2,wo,c1,c2
		real         :: rfix1,rfix2,farea1,farea2
		real         :: ropen1,ropen2,alegg1,blegg1,&
		                alegg2,blegg2,theta,dp1,dp2,w,wqudlin,R1,R2,RDAMPER2

        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        
!        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
!                                 blegg=(/0.105,0.0842/)
	    real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)
!	    real,dimension(0:1)   :: alegg=(/-4.011,-1.1811/),&
!                                 blegg=(/0.1983,0.1184/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,0.4429/),&
!                                 blegg=(/0.1945,0.0679/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,-0.0261/),&
!                                 blegg=(/0.1945,0.078/)
!         real,dimension(0:1)   :: alegg=(/-4.011,-1.6325/),&
!                                  blegg=(/0.1983,0.1181/)

! **** Loss coefficient of fully open damper
        real         :: kopenh=1.1149,kopenc=2.0368
!         real         :: kopenh=1.1149,kopenc=3.3404

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakh=0.0149,fleakc=0.0096
!		real         :: fleakh=0.0149,fleakc=0.0076

! **** Assume opposed or single blade damper
!        integer      :: ipx1=0,ipx2=0,i
		integer      :: ipx1=0,ipx2=1,i

! **** Read in inputs
        PI1=XIN(1)
        PI2=XIN(2)
        PO=XIN(3)
	    C1=XIN(4)
	    C2=XIN(5)
! **** Read and check parameters
      
        RFIX1= PAR_V(1)
!        IF (RFIX1.LE.0.0) STOP 'TYPE535: RFIX1 NOT GREATER THAN 0'
        RFIX2= PAR_V(2)
!        IF (RFIX2.LE.0.0) STOP 'TYPE535: RFIX2 NOT GREATER THAN 0'
	    
        FAREA1= PAR_V(3)
        FAREA2= PAR_V(4)
               
! **** Damper resistance
! **** Resistance when fully open

        ROPEN1 = 1.0E-3*KOPENH/(2.0*RHOA*FAREA1*FAREA1)
        ROPEN2 = 1.0E-3*KOPENC/(2.0*RHOA*FAREA2*FAREA2)

! **** Resistance at position C1 or C2
        ALEGG1 = ALEGG(0)                 
        BLEGG1 = BLEGG(0)
		
		ALEGG2 = ALEGG(1)                 
        BLEGG2 = BLEGG(1)    
               	               
        R1 = RDAMPER2(C1,ROPEN1,FLEAKH,FAREA1,ALEGG1,BLEGG1,IPX1)+RFIX1
	    R2 = RDAMPER2(C2,ROPEN2,FLEAKC,FAREA2,ALEGG2,BLEGG2,IPX2)+RFIX2

       
! **** Calculate mass flow rate 1
        DP1 = PI1-PO
        WI1 = WQUDLIN(R1,DP1)

! **** Calculate mass flow rate 2
	    DP2 = PI2-PO
	    WI2 = WQUDLIN(R2,DP2)

! **** Calculate total mass flow rate
	    WO = WI1+WI2
! **** Assign outputs
        YOUT(1)=WI1
        YOUT(2)=WI2
        YOUT(3)=WO
! **** Allow freezing of algebraic variables
        do i=1,no
                iostat(i) = 1
        enddo
        RETURN
        END SUBROUTINE TYPE535
!**********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! *
! * SUBROUTINE: Motorized pressure-independent dual duct box 2 
! *
! * PURPOSE:     Calculate the outlet flow rate and the inlet pressures 
! *              for a VAV mixing box, given the outlet pressure and the
! *              inlet flow rates
! *
!***********************************************************************
! * INPUTS
! * ======
! *  1. PIH     : hot deck inlet pressure                          (kPa)
! *  2. WIC     : cold deck inlet flow rate                       (kg/s)
! *  3. PO      : pressure at outlet                               (kPa)
! *  4. CHOT    : hot deck damper position (0=closed, 1=open)        (-)
! *  5. CCOLD   : cold deck damper position (0=closed, 1=open)       (-)
! *
! * OUTPUTS
! * =======
! *  1. WIH     : hot deck inlet air flow rate                    (kg/s)
! *  2. PIC     : cold deck inlet air pressure                     (kPa)
! *  3. WO      : outlet air flow rate                            (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. RFIXH   : hot deck fixed resistance                  (0.001/k.m)
! *  2. RFIXC   : cold deck fixed resistance                 (0.001/k.m)
! *  3. FAREAH  : Face area of hot damper                           (m2)
! *  4. FAREAC  : Face area of cold damper                          (m2)
!***********************************************************************
!
!   MAJOR RESTRICTIONS:  None
!
!   DEVELOPER:           XIWANG LI
!
!   DATE:                
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWMERG
!   FUNCTIONS  CALLED:   RDAMPER
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
!***********************************************************************

        subroutine type536(xin,yout,par_v,saved_v,iostat)
	    use modsim_head
        implicit none
        integer,parameter                 :: ni=5,no=3,np=4,ns=0
		real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: pih,pic,po,wih,wic,wo,chot,ccold
		real         :: rfixh,rfixc,fareah,fareac
		real         :: ropenh,ropenc,alegg1,blegg1,&
		                alegg2,blegg2,theta,dpqudlin,dph,wqudlin,RH,RC,RDAMPER2

        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        
!        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
!                                 blegg=(/0.105,0.0842/)
	    real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)
!	    real,dimension(0:1)   :: alegg=(/-4.011,-1.1811/),&
!                                 blegg=(/0.1983,0.1184/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,0.4429/),&
!                                 blegg=(/0.1945,0.0679/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,-0.0261/),&
!                                 blegg=(/0.1945,0.078/)
!         real,dimension(0:1)   :: alegg=(/-4.011,-1.6325/),&
!                                  blegg=(/0.1983,0.1181/)

! **** Loss coefficient of fully open damper
        real         :: kopenh=1.1149,kopenc=2.0368
!         real         :: kopenh=1.1149,kopenc=3.3404

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakh=0.0149,fleakc=0.0096
!		real         :: fleakh=0.0149,fleakc=0.0076

! **** Assume opposed or single blade damper
!        integer      :: ipxh=0,ipxc=0,i
		integer      :: ipxh=0,ipxc=1,i

! **** Read in inputs
        PIH   = XIN(1)
        WIC   = XIN(2)
        PO    = XIN(3)
 	    CHOT  = XIN(4)
	    CCOLD = XIN(5)

! **** Read and check parameters
      
        RFIXH= PAR_V(1)
!        IF (RFIXH.LE.0.0) STOP 'TYPE536: RFIXH NOT GREATER THAN 0'
       
        RFIXC= PAR_V(2)
!        IF (RFIXC.LE.0.0) STOP 'TYPE536: RFIXC NOT GREATER THAN 0'
	    
        FAREAH= PAR_V(3)
        FAREAC= PAR_V(4)
!		open (unit = 78, file = 'HOTDAMPER.txt')
!		open (unit = 79, file = 'COLDDAMPER.txt')
               
! **** Damper resistance
! **** Resistance when fully open

        ROPENH = 1.0E-3*KOPENH/(2.0*RHOA*FAREAH*FAREAH)
        ROPENC = 1.0E-3*KOPENC/(2.0*RHOA*FAREAC*FAREAC)

! **** Resistance at position C1 or C2
        ALEGG1 = ALEGG(0)                 
        BLEGG1 = BLEGG(0)
		
    	ALEGG2 = ALEGG(1)                 
        BLEGG2 = BLEGG(1)    
               	               
        RH = RDAMPER2(CHOT,ROPENH,FLEAKH,FAREAH,ALEGG1,BLEGG1,IPXH)+RFIXH
!		 write (78, *) time/60,'	',CHOT,'	',RH

        RC = RDAMPER2(CCOLD,ROPENC,FLEAKC,FAREAC,ALEGG2,BLEGG2,IPXC)+RFIXC
!		 write (79, *) time/60,'	',CCOLD,'	',RC
       
! **** Calculate hot deck flow
        DPH = PIH-PO
        WIH = WQUDLIN(RH,DPH)

! **** Calculate cold deck pressure
        
        PIC = PO+DPQUDLIN(RC,WIC)

! **** Calculate total flow
   
       WO = WIH+WIC
        
! **** Assign outputs
        
        YOUT(1) = WIH
        YOUT(2) = PIC
        YOUT(3) = WO
! **** Allow freezing of algebraic variables
        do i=1,no
                iostat(i) = 1
        enddo
        RETURN
        END SUBROUTINE TYPE536
!**********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
!***********************************************************************
! *
! * SUBROUTINE: Motorized pressure-independent dual duct box 2 
! *
! * PURPOSE:     Calculate the outlet flow rate and the inlet pressures 
! *              for a VAV mixing box, given the outlet pressure and the
! *              inlet flow rates
! *
!***********************************************************************
! * INPUTS
! * ======
! *  1. PIC     : cold deck inlet pressure                         (kPa)
! *  2. WIH     : hot deck inlet flow rate                        (kg/s)
! *  3. PO      : pressure at outlet                               (kPa)
! *  4. CHOT    : hot deck damper position (0=closed, 1=open)        (-)
! *  5. CCOLD   : cold damper position (0=closed, 1=open)            (-)
! *
! * OUTPUTS
! * =======
! *  1. WIC     : cold deck inlet air flow rate                   (kg/s)
! *  2. PIH     : hot deck inlet air pressure                      (kPa)
! *  3. WO      : outlet air flow rate                            (kg/s)
! *
! * PARAMETERS
! * ==========
! *  1. RFIXH   : hot deck fixed resistance                  (0.001/k.m)
! *  2. RFIXC   : cold deck fixed resistance                 (0.001/k.m)
! *  3. FAREAH  : Face area of hot damper                           (m2)
! *  4. FAREAC  : Face area of cold damper                          (m2)
!***********************************************************************
!
!   MAJOR RESTRICTIONS:  None
!
!   DEVELOPER:           XIWANG LI
!
!   DATE:                
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  FLOWMERG
!   FUNCTIONS  CALLED:   RDAMPER
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
!***********************************************************************

        subroutine type538(xin,yout,par_v,saved_v,iostat)
	    use modsim_head
        implicit none
        integer,parameter                 :: ni=5,no=3,np=4,ns=0
		real,dimension(ni),intent(in)     :: xin
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: pih,pic,po,wih,wic,wo,chot,ccold
		real         :: rfixh,rfixc,fareah,fareac
		real         :: ropenh,ropenc,alegg1,blegg1,&
		                alegg2,blegg2,theta,dpqudlin,dpc,wqudlin,RH,RC,RDAMPER2

        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
       
!        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
!                                 blegg=(/0.105,0.0842/)
	    real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)
!	    real,dimension(0:1)   :: alegg=(/-4.011,-1.1811/),&
!                                 blegg=(/0.1983,0.1184/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,0.4429/),&
!                                 blegg=(/0.1945,0.0679/)
!        real,dimension(0:1)   :: alegg=(/-4.4941,-0.0261/),&
!                                 blegg=(/0.1945,0.078/)
!         real,dimension(0:1)   :: alegg=(/-4.011,-1.6325/),&
!                                  blegg=(/0.1983,0.1181/)

! **** Loss coefficient of fully open damper
        real         :: kopenh=1.1149,kopenc=2.0368
!         real         :: kopenh=1.1149,kopenc=3.3404

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakh=0.0149,fleakc=0.0096
!		real         :: fleakh=0.0149,fleakc=0.0076

! **** Assume opposed or single blade damper
!        integer      :: ipxh=0,ipxc=0,i
		integer      :: ipxh=0,ipxc=1,i

! **** Read in inputs
        PIC   = XIN(1)
        WIH   = XIN(2)
        PO    = XIN(3)
 	    CHOT  = XIN(4)
	    CCOLD = XIN(5)

! **** Read and check parameters
      
        RFIXH= PAR_V(1)
 !       IF (RFIXH.LE.0.0) STOP 'TYPE537: RFIXH NOT GREATER THAN 0'
       
        RFIXC= PAR_V(2)
 !       IF (RFIXC.LE.0.0) STOP 'TYPE537: RFIXC NOT GREATER THAN 0'
	    
        FAREAH= PAR_V(3)
        FAREAC= PAR_V(4)
               
! **** Damper resistance
! **** Resistance when fully open

        ROPENH = 1.0E-3*KOPENH/(2.0*RHOA*FAREAH*FAREAH)
        ROPENC = 1.0E-3*KOPENC/(2.0*RHOA*FAREAC*FAREAC)

! **** Resistance at position C1 or C2
        ALEGG1 = ALEGG(0)                 
        BLEGG1 = BLEGG(0)
		
    	ALEGG2 = ALEGG(1)                 
        BLEGG2 = BLEGG(1)    
               	               
        RH = RDAMPER2(CHOT,ROPENH,FLEAKH,FAREAH,ALEGG1,BLEGG1,IPXH)+RFIXH

        RC = RDAMPER2(CCOLD,ROPENC,FLEAKC,FAREAC,ALEGG2,BLEGG2,IPXC)+RFIXC
       
! **** Calculate cold deck flow
        DPC = PIC-PO
        WIC = WQUDLIN(RC,DPC)

! **** Calculate hot deck pressure
        
        PIH = PO+DPQUDLIN(RH,WIH)

! **** Calculate total flow
   
       WO = WIH+WIC
        
! **** Assign outputs
        
        YOUT(1) = WIC
        YOUT(2) = PIH
        YOUT(3) = WO
! **** Allow freezing of algebraic variables
        do i=1,no
                iostat(i) = 1
        enddo
        RETURN
        END SUBROUTINE TYPE538
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent dual duct VAV box1
! *
! * PURPOSE:        Calculate the inlet pressure of a motor-driven,
! *                 pressure-independent VAV box from the outlet
! *                 pressure, the mass flow rate and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wairc    : air mass flow rate of cold deck                 (kg/s)
! *  2. pinh     : inlet pressure of hot deck                       (kpa)
! *  3. pout     : outlet pressure                                  (kPa)
! *  4. fflowsetc: demanded flowrate of cold deck                   (0-1)
! *  5. fflowseth: demanded flowrate of hot deck                    (0-1)
! *  6. wairh    : air mass flow rate of hot deck                  (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. wairh    : air mass flow rate of hot deck                  (kg/s)
! *  2. pinc     : inlet pressure of cold deck                      (kPa)
! *  3. wout     : mass flow rate at box outlet                    (kg/s)
! *  4. dampposc : damper position of cold deck (0=closed, 1=open)    (-)
! *  5. dampposh : damper position of hot deck (0=closed, 1=open)     (-)
! *  6. fspedc   : fractional motor velocity of cold deck (0-1)       (-)
! *  7. fspedh   : fractional motor velocity of hot deck (0-1)        (-)
! *  8. tssrevc  : number of stops/starts/reversals of cold deck      (-)
! *  9. tssrevh  : number of stops/starts/reversals of hot deck       (-)
! * 10. ttravc   : total distance travelled by valve/damper cold deck (-)
! * 11. ttravh   : total distance travelled by valve/damper hot deck  (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnomc: nominal volumetric flow rate of cold deck        (m3/s)                       
! *  2. volumnomh: nominal volumetric flow rate of hot deck         (m3/s)                       
! *  3. fareac   : face area of damper of cold deck                   (m2)                                
! *  4. fareah   : face area of damper of hot deck                    (m2)                                
! *  5. rboxc    : cold deck dual duct box resistnace         (1/1000kg.m)               
! *  6. rboxh    : hot deck dual duct box resistnace          (1/1000kg.m)              
! *  7. ttranc   : actuator travel time (0-90 deg) of cold deck        (s)                       
! *  8. ttranh   : actuator travel time (0-90 deg) of hot deck         (s)             
! *  9. fspedminc: minimum fractional motor speed cold deck            (-)                              
! * 10. fspedminh: minimum fractional motor speed hot deck             (-)                              
! * 11. hysc     : hysteresis of cold deck                             (-)                                               
! * 12. hysh     : hysteresis of hot deck                              (-)                                             
! * 13. kpc      : controller gain (frac speed/frac error) of cold deck(-)                    
! * 14. kph      : controller gain (frac speed/frac error) of hot deck (-)  
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actposc  : actuator position at previous call
! *  3. actpospc : actuator position at previous time-step
! *  4. fspedc   : actuator speed at previous call
! *  5. fspedpc  : actuator speed at prev time-step
! *  6. ispedc   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedpc  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrevc  : total no. of stop/starts/reversals at previous call
! *  9. tssrevpc : total no. of stop/starts/reversals at prev time-step
! * 10. ttravc   : total distance travelled at previous call
! * 11. ttravpc  : total distance travelled at previous timestep
! * 12. dampposc : position of final control element at previous call
! * 13. damppospc: position of final control element at previous step time
! * 14. actposh  : actuator position at previous call
! * 15. actposph : actuator position at previous time-step
! * 16. fspedh   : actuator speed at previous call
! * 17. fspedph  : actuator speed at prev time-step
! * 18. ispedh   : +1=forward, -1=backward, 0=stopped at previous call
! * 19. ispedph  : +1=forward, -1=backward, 0=stopped at prev time-step
! * 20. tssrevh  : total no. of stop/starts/reversals at previous call
! * 21. tssrevph : total no. of stop/starts/reversals at prev time-step
! * 22. ttravh   : total distance travelled at previous call
! * 23. ttravph  : total distance travelled at previous timestep
! * 24. dampposh : position of final control element at previous call
! * 25. dampposph: position of final control element at previous step time

! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian,Jin Wen
!                        Drexel University
!                        Philip Haves
!                        Loughborough University of Technology
!                  
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
! **********************************************************************
!
!   Updated to Fortran 90 Augest 2, 2012  Xiwang LI, Drexel University
!
! **********************************************************************
       
	    subroutine TYPE531(xin,yout,par_v,saved_v,iostat)

	    use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=6,no=11,np=14,&
                                             ns=25
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
       
	    real         :: rhoa=1.2
        logical      :: con

! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)


! **** Loss coefficient of fully open damper
        real         :: kopenc = 2.0368, kopenh = 1.1149

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakc = 0.0096, fleakh = 0.0149

! **** Assume damper is fully open at start
        real         :: startpos=1.0

        real         :: volumnomc,fareac,rboxc,ttranc,fspedminc,hysc,&
                        fspedpc,tssrevpc,ttravpc,damppospc,fflowc,&
                        fspedc,fspedbarc,actposc,tssrevc,ttravc,dampposc,&
                        aleggc,bleggc,rc,raddc,rtotc,pinc,wairc,pout,&
                        actpospc, &
						ropenc,fflowsetc,fspeddemc,kpc
	    real         :: volumnomh,fareah,rboxh,ttranh,fspedminh,hysh,&
                        fspedph,tssrevph,ttravph,dampposph,fflowh,&
                        fspedh,fspedbarh,actposh,tssrevh,ttravh,dampposh,&
                        aleggh,bleggh,rh,raddh,rtoth,pinh,wairh,&
                        spedlim,clip,rdamper,dpqudlin,actposph, &
						ropenh,fflowseth,fspeddemh,kph,wout,rout,rdamper2,wqudlin,dph
					
        integer      :: i,is,ispedpc,ispedc,ispedph,ispedh

! **** Assume opposed or single blade damper
        integer      :: ipc = 1, iph = 0

! **** Read in inputs
        wairc     = xin(1)
        pinh      = xin(2)
        pout      = xin(3)
        fflowsetc = xin(4)
        fflowseth = xin(5)
		wairh     = xin(6)
! **** Read and check parameters
        
        volumnomc = par_v(1)

        if (volumnomc<=0.0) then
            stop 'type 531: cold deck nominal flow rate must be greater than zero'
        endif
      
	   volumnomh = par_v(2)
        if (volumnomh<=0.0) then
            stop 'type 531: hot deck nominal flow rate must be greater than zero'
        endif

        fareac    = par_v(3)
		fareah    = par_v(4)
        rboxc     = par_v(5)
        rboxh     = par_v(6)
        ttranc    = par_v(7)
        ttranh    = par_v(8)
        fspedminc = par_v(9)
		fspedminh = par_v(10)
		hysc      = par_v(11)
		hysh      = par_v(12)
        kpc       = par_v(13)
		kph       = par_v(14)
		
! **** Initialize at beginning of simulation
        if (itime<=1) then 
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
				saved_v(14) = startpos
                do is = 4,10,2
                    saved_v(is) = 0.0
                enddo
				do is = 16,22,2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
				saved_v(24) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif

! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actpospc  = saved_v(3)
        fspedpc   = saved_v(5)
        ispedpc   = saved_v(7)
        tssrevpc  = saved_v(9)
        ttravpc   = saved_v(11)
        damppospc = saved_v(13)
		actposph  = saved_v(15)
        fspedph   = saved_v(17)
        ispedph   = saved_v(19)
        tssrevph  = saved_v(21)
        ttravph   = saved_v(23)
        dampposph = saved_v(25)

! **** Limit control signal
        if (fflowsetc<0.0) fflowsetc = 0.0
        if (fflowsetc>1.0) fflowsetc = 1.0
        if (fflowseth<0.0) fflowseth = 0.0
        if (fflowseth>1.0) fflowseth = 1.0

! **** Measured fractional flow at end of time-step
        fflowc    = (wairc/rhoa)/volumnomc
		fflowh    = (wairh/rhoa)/volumnomh
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddemc = kpc*(fflowsetc-fflowc)
        fspedc    = spedlim(fspeddemc,fspedminc)

		fspeddemh = kph*(fflowseth-fflowh)
        fspedh    = spedlim(fspeddemh,fspedminh)
      
 ! **** Average speed over time-step
        fspedbarc = (fspedc+fspedpc)/2.0     
        fspedbarh = (fspedh+fspedph)/2.0     
! **** New actuator position
        actposc   = actpospc+fspedbarc*tstep/ttranc
        actposc   = clip(actposc,0.0,1.0)
		actposh   = actposph+fspedbarh*tstep/ttranh
        actposh   = clip(actposh,0.0,1.0)
!		if (actposh<=0.3) then
!		    actposh=actposh+0.2
!        endif
! **** Fault flag on
        if (CLGdamp==0) then
		    actposc = actposc
        elseif (CLGdamp==1) then
		    actposc = VCLGdamp
		endif

		if (HTGdamp==0) then
		    actposh = actposh
        elseif (HTGdamp==1) then
		    actposh = VHTGdamp
		endif
! **** Fault flag off

! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fspedc>0.0) then
            ispedc = +1
        elseif (fspedc<0.0) then
            ispedc = -1
        else
            ispedc = 0
        endif

		if (fspedh>0.0) then
            ispedh = +1
        elseif (fspedh<0.0) then
            ispedh = -1
        else
            ispedh = 0
        endif

! **** Previous stops/starts/reversals
        if (ispedc/=ispedpc) then
            tssrevc = tssrevpc+1.0
        else
            tssrevc = tssrevpc
        endif

		 if (ispedh/=ispedph) then
            tssrevh = tssrevph+1.0
        else
            tssrevh = tssrevph
        endif
! **** Distance travelled
        ttravc = ttravpc+abs(actposc-actpospc)
		ttravh = ttravph+abs(actposh-actposph)

! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actposc,damppospc,hysc,dampposc)
		call hystrsis(actposh,dampposph,hysh,dampposh)

!		If (dampposh <=0.29) dampposh=0.49

! **** Resistance when fully open
        ropenc = 1.0e-3*kopenc/(2.0*rhoa*fareac*fareac)
		ropenh = 1.0e-3*kopenh/(2.0*rhoa*fareah*fareah)

! **** Resistance at position damppos
        aleggc = alegg(0)                 
        bleggc = blegg(0)                
		aleggh = alegg(1)                
        bleggh = blegg(1)                
        rc = rdamper2(dampposc,ropenc,fleakc,fareac,aleggc,bleggc,ipc)
		rh = rdamper2(dampposh,ropenh,fleakh,fareah,aleggh,bleggh,iph)
! **** Resistance of box, diffuser etc, excluding damper
!        raddc = pdopenc/(rhoa*volumnomc*rhoa*volumnomc) - ropenc
        rtotc = rc+rboxc
!		raddh = pdopenh/(rhoa*volumnomh*rhoa*volumnomh) - ropenh
        rtoth = rh+rboxh
! **** Pressure drop and inlet pressure for cold deck
        pinc = pout+dpqudlin(rtotc,wairc)

! **** Calculate mass flow rate for hot deck 
        dph  = pinh-pout
		wairh= wqudlin(rtoth,dph)
		wout=wairc+wairh
           
! **** Save time of current call
         saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actposc
        saved_v(4)  = fspedc
        saved_v(6)  = float(ispedc)
        saved_v(8)  = tssrevc
        saved_v(10) = ttravc
        saved_v(12) = dampposc
        saved_v(14) = actposh
        saved_v(16) = fspedh
        saved_v(18) = float(ispedh)
        saved_v(20) = tssrevh
        saved_v(22) = ttravh
        saved_v(24) = dampposh
! **** Output
        yout(1)  = wairh
		yout(2)  = pinc
		yout(3)  = wout
        yout(4)  = dampposc
		yout(5)  = dampposh
        yout(6)  = fspedc
		yout(7)  = fspedh
        yout(8)  = tssrevc
		yout(9)  = tssrevh
        yout(10) = ttravc
		yout(11) = ttravh

! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbarc==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type531
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent dual duct VAV box3
! *
! * PURPOSE:        Calculate the inlet pressure of a motor-driven,
! *                 pressure-independent VAV box from the outlet
! *                 pressure, the mass flow rate and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wairh    : air mass flow rate of hot deck                  (kg/s)
! *  2. pinc     : inlet pressure of cold deck                      (kpa)
! *  3. pout     : outlet pressure                                  (kPa)
! *  4. fflowsetc: demanded flowrate of cold deck                   (0-1)
! *  5. fflowseth: demanded flowrate of hot deck                    (0-1)
! *  6. wairc    : air mass flow rate of cold deck                 (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. wairc    : air mass flow rate of cold deck                 (kg/s)
! *  2. pinh     : inlet pressure of hot deck                       (kPa)
! *  3. wout     : mass flow rate at box outlet                    (kg/s)
! *  4. dampposc : damper position of cold deck (0=closed, 1=open)    (-)
! *  5. dampposh : damper position of hot deck (0=closed, 1=open)     (-)
! *  6. fspedc   : fractional motor velocity (0-1)  of cold deck      (-)
! *  7. fspedh   : fractional motor velocity (0-1)  of hot deck       (-)
! *  8. tssrevc  : number of stops/starts/reversals of cold deck      (-)
! *  9. tssrevh  : number of stops/starts/reversals of hot deck       (-)
! * 10. ttravc   : total distance travelled by valve/damper cold deck (-)
! * 11. ttravh   : total distance travelled by valve/damper hot deck  (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnomc: nominal volumetric flow rate of cold deck        (m3/s)                       
! *  2. volumnomh: nominal volumetric flow rate of hot deck         (m3/s)                       
! *  3. fareac   : face area of damper of cold deck                   (m2)                                
! *  4. fareah   : face area of damper of hot deck                    (m2)                                
! *  5. rboxc    : cold deck dual duct box resistnace         (1/1000kg.m)               
! *  6. rboxh    : hot deck dual duct box resistnace          (1/1000kg.m)                  
! *  7. ttranc   : actuator travel time (0-90 deg) of cold deck        (s)                       
! *  8. ttranh   : actuator travel time (0-90 deg) of hot deck         (s)             
! *  9. fspedminc: minimum fractional motor speed cold deck            (-)                              
! * 10. fspedminh: minimum fractional motor speed hot deck             (-)                              
! * 11. hysc     : hysteresis of cold deck                             (-)                                               
! * 12. hysh     : hysteresis of hot deck                              (-)                                             
! * 13. kpc      : controller gain (frac speed/frac error) of cold deck(-)                    
! * 14. kph      : controller gain (frac speed/frac error) of hot deck (-)  
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actposc  : actuator position at previous call
! *  3. actpospc : actuator position at previous time-step
! *  4. fspedc   : actuator speed at previous call
! *  5. fspedpc  : actuator speed at prev time-step
! *  6. ispedc   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedpc  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrevc  : total no. of stop/starts/reversals at previous call
! *  9. tssrevpc : total no. of stop/starts/reversals at prev time-step
! * 10. ttravc   : total distance travelled at previous call
! * 11. ttravpc  : total distance travelled at previous timestep
! * 12. dampposc : position of final control element at previous call
! * 13. damppospc: position of final control element at previous step time
! * 14. actposh  : actuator position at previous call
! * 15. actposph : actuator position at previous time-step
! * 16. fspedh   : actuator speed at previous call
! * 17. fspedph  : actuator speed at prev time-step
! * 18. ispedh   : +1=forward, -1=backward, 0=stopped at previous call
! * 19. ispedph  : +1=forward, -1=backward, 0=stopped at prev time-step
! * 20. tssrevh  : total no. of stop/starts/reversals at previous call
! * 21. tssrevph : total no. of stop/starts/reversals at prev time-step
! * 22. ttravh   : total distance travelled at previous call
! * 23. ttravph  : total distance travelled at previous timestep
! * 24. dampposh : position of final control element at previous call
! * 25. dampposph: position of final control element at previous step time

! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian,Jin Wen
!                        Drexel University
!                        Philip Haves
!                        Loughborough University of Technology
!                  
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
! **********************************************************************
!
!   Updated to Fortran 90 Augest 2, 2012  Xiwang LI, Drexel University
!
! **********************************************************************
       
	    subroutine TYPE534(xin,yout,par_v,saved_v,iostat)

	    use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=6,no=11,np=14,&
                                             ns=25
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
       
	    real         :: rhoa=1.2
        logical      :: con

! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)


! **** Loss coefficient of fully open damper
        real         :: kopenc = 2.0368, kopenh = 1.1149

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakc = 0.0096, fleakh = 0.0149

! **** Assume damper is fully open at start
        real         :: startpos=1.0

        real         :: volumnomc,fareac,rboxc,ttranc,fspedminc,hysc,&
                        fspedpc,tssrevpc,ttravpc,damppospc,fflowc,&
                        fspedc,fspedbarc,actposc,tssrevc,ttravc,dampposc,&
                        aleggc,bleggc,rc,raddc,rtotc,pinc,wairc,pout,&
                        actpospc, &
						ropenc,fflowsetc,fspeddemc,kpc
	    real         :: volumnomh,fareah,rboxh,ttranh,fspedminh,hysh,&
                        fspedph,tssrevph,ttravph,dampposph,fflowh,&
                        fspedh,fspedbarh,actposh,tssrevh,ttravh,dampposh,&
                        aleggh,bleggh,rh,raddh,rtoth,pinh,wairh,&
                        spedlim,clip,rdamper,dpqudlin,actposph, &
						ropenh,fflowseth,fspeddemh,kph,wout,rout,rdamper2,wqudlin,dpc
					
        integer      :: i,is,ispedpc,ispedc,ispedph,ispedh

! **** Assume opposed or single blade damper
        integer      :: ipc = 1, iph = 0

! **** Read in inputs
        wairh     = xin(1)
        pinc      = xin(2)
        pout      = xin(3)
        fflowsetc = xin(4)
        fflowseth = xin(5)
		wairc     = xin(6)
! **** Read and check parameters
        
        volumnomc = par_v(1)

        if (volumnomc<=0.0) then
            stop 'type 534: cold deck nominal flow rate must be greater than zero'
        endif
      
	    volumnomh = par_v(2)
        if (volumnomh<=0.0) then
            stop 'type 534: hot deck nominal flow rate must be greater than zero'
        endif

        fareac    = par_v(3)
		fareah    = par_v(4)
        rboxc     = par_v(5)
        rboxh     = par_v(6)
        ttranc    = par_v(7)
        ttranh    = par_v(8)
        fspedminc = par_v(9)
		fspedminh = par_v(10)
		hysc      = par_v(11)
		hysh      = par_v(12)
        kpc       = par_v(13)
		kph       = par_v(14)
		
! **** Initialize at beginning of simulation
        if (itime<=1) then 
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
				saved_v(14) = startpos
                do is = 4,10,2
                    saved_v(is) = 0.0
                enddo
				do is = 16,22,2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
				saved_v(24) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif

! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actpospc  = saved_v(3)
        fspedpc   = saved_v(5)
        ispedpc   = saved_v(7)
        tssrevpc  = saved_v(9)
        ttravpc   = saved_v(11)
        damppospc = saved_v(13)
		actposph  = saved_v(15)
        fspedph   = saved_v(17)
        ispedph   = saved_v(19)
        tssrevph  = saved_v(21)
        ttravph   = saved_v(23)
        dampposph = saved_v(25)

! **** Limit control signal
        if (fflowsetc<0.0) fflowsetc = 0.0
        if (fflowsetc>1.0) fflowsetc = 1.0
        if (fflowseth<0.0) fflowseth = 0.0
        if (fflowseth>1.0) fflowseth = 1.0

! **** Measured fractional flow at end of time-step
        fflowc    = (wairc/rhoa)/volumnomc
		fflowh    = (wairh/rhoa)/volumnomh
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddemc = kpc*(fflowsetc-fflowc)
        fspedc    = spedlim(fspeddemc,fspedminc)

		fspeddemh = kph*(fflowseth-fflowh)
        fspedh    = spedlim(fspeddemh,fspedminh)
      
 ! **** Average speed over time-step
        fspedbarc = (fspedc+fspedpc)/2.0     
        fspedbarh = (fspedh+fspedph)/2.0     
! **** New actuator position
        actposc   = actpospc+fspedbarc*tstep/ttranc
        actposc   = clip(actposc,0.0,1.0)
		actposh   = actposph+fspedbarh*tstep/ttranh
        actposh   = clip(actposh,0.0,1.0)
!	    if (actposh<=0.3) then
!		    actposh=actposh+0.2
!        endif

! **** Fault flag on
        if (CLGdamp==0) then
		    actposc = actposc
        elseif (CLGdamp==1) then
		    actposc = VCLGdamp
		endif

		if (HTGdamp==0) then
		    actposh = actposh
        elseif (HTGdamp==1) then
		    actposh = VHTGdamp
		endif
! **** Fault flag off

! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fspedc>0.0) then
            ispedc = +1
        elseif (fspedc<0.0) then
            ispedc = -1
        else
            ispedc = 0
        endif

		if (fspedh>0.0) then
            ispedh = +1
        elseif (fspedh<0.0) then
            ispedh = -1
        else
            ispedh = 0
        endif

! **** Previous stops/starts/reversals
        if (ispedc/=ispedpc) then
            tssrevc = tssrevpc+1.0
        else
            tssrevc = tssrevpc
        endif

		 if (ispedh/=ispedph) then
            tssrevh = tssrevph+1.0
        else
            tssrevh = tssrevph
        endif
! **** Distance travelled
        ttravc = ttravpc+abs(actposc-actpospc)
		ttravh = ttravph+abs(actposh-actposph)

! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actposc,damppospc,hysc,dampposc)
		call hystrsis(actposh,dampposph,hysh,dampposh)

!		If (dampposh <=0.29) dampposh=0.49

! **** Resistance when fully open
        ropenc = 1.0e-3*kopenc/(2.0*rhoa*fareac*fareac)
		ropenh = 1.0e-3*kopenh/(2.0*rhoa*fareah*fareah)

! **** Resistance at position damppos
        aleggc = alegg(0)                 
        bleggc = blegg(0)                
		aleggh = alegg(1)                
        bleggh = blegg(1)                
        rc = rdamper2(dampposc,ropenc,fleakc,fareac,aleggc,bleggc,ipc)
		rh = rdamper2(dampposh,ropenh,fleakh,fareah,aleggh,bleggh,iph)
! **** Resistance of box, diffuser etc, excluding damper
!        raddc = pdopenc/(rhoa*volumnomc*rhoa*volumnomc) - ropenc
        rtotc = rc+rboxc
!		raddh = pdopenh/(rhoa*volumnomh*rhoa*volumnomh) - ropenh
        rtoth = rh+rboxh
! **** Pressure drop and inlet pressure for hot deck
        pinh = pout+dpqudlin(rtoth,wairh)

! **** Calculate mass flow rate for cold deck 
        dpc  = pinc-pout
		wairc= wqudlin(rtotc,dpc)
		wout=wairc+wairh
           
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actposc
        saved_v(4)  = fspedc
        saved_v(6)  = float(ispedc)
        saved_v(8)  = tssrevc
        saved_v(10) = ttravc
        saved_v(12) = dampposc
        saved_v(14) = actposh
        saved_v(16) = fspedh
        saved_v(18) = float(ispedh)
        saved_v(20) = tssrevh
        saved_v(22) = ttravh
        saved_v(24) = dampposh
! **** Output
        yout(1)  = wairc
		yout(2)  = pinh
		yout(3)  = wout
        yout(4)  = dampposc
		yout(5)  = dampposh
        yout(6)  = fspedc
		yout(7)  = fspedh
        yout(8)  = tssrevc
		yout(9)  = tssrevh
        yout(10) = ttravc
		yout(11) = ttravh

! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbarc==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type534

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent dual duct VAV box2
! *                   - calculates flow 
! *
! * PURPOSE:        Calculate the inlet pressure of a motor-driven,
! *                 pressure-independent VAV box from the outlet
! *                 pressure, the mass flow rate and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. pinc     : inlet pressure of cold deck                       (kPa)
! *  2. pinh     : inlet pressure of hot deck                        (kPa)
! *  3. pout     : outlet pressure                                   (kPa)
! *  4. fflowsetc: demanded flowrate of cold deck                    (0-1)
! *  5. fflowseth: demanded flowrate of hot deck                     (0-1)
! *  6. wairc    : air mass flow rate of cold deck                  (kg/s)
! *  7. wairh    : air mass flow rate of hot deck                   (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. wairc    : air mass flow rate of cold deck                  (kg/s)
! *  2. wairh    : air mass flow rate of hot deck                   (kg/s)
! *  3. wout     : air mass flow rate of box outlet                 (kg/s)
! *  4. dampposc : damper position of cold deck (0=closed, 1=open)     (-)
! *  5. dampposh : damper position of hot deck (0=closed, 1=open)      (-)
! *  6. fspedc   : fractional motor velocity (0-1)  of cold deck       (-)
! *  7. fspedh   : fractional motor velocity (0-1)  of hot deck        (-)
! *  8. tssrevc  : number of stops/starts/reversals of cold deck       (-)
! *  9. tssrevh  : number of stops/starts/reversals of hot deck        (-)
! * 10. ttravc   : total distance travelled by valve/damper cold deck  (-)
! * 11. ttravh   : total distance travelled by valve/damper hot deck   (-)
! *
! * PARAMETERS
! * ==========
! *  1. volumnomc: nominal volumetric flow rate of cold deck        (m3/s)                       
! *  2. volumnomh: nominal volumetric flow rate of hot deck         (m3/s)                       
! *  3. fareac   : face area of damper(s) of cold deck                (m2)                                
! *  4. fareah   : face area of damper(s) of hot deck                 (m2)                                
! *  5. rboxc    : cold deck dual duct box resistnace         (1/1000kg.m)               
! *  6. rboxh    : hot deck dual duct box resistnace          (1/1000kg.m)                 
! *  7. ttranc   : actuator travel time (0-90 deg) of cold deck        (s)                       
! *  8. ttranh   : actuator travel time (0-90 deg) of hot deck         (s)             
! *  9. fspedminc: minimum fractional motor speed cold deck            (-)                              
! * 10. fspedminh: minimum fractional motor speed hot deck             (-)                              
! * 11. hysc     : hysteresis of cold deck                             (-)                                               
! * 12. hysh     : hysteresis of hot deck                              (-)                                             
! * 13. kpc      : controller gain (frac speed/frac error) of cold deck(-)                    
! * 14. kph      : controller gain (frac speed/frac error) of hot deck (-)  
! * 15. location : 1- room SA,2- room SB                               (-)
! *
! *  SAVED
! *  =====
! *  1. time     : time of previous call for rate limit
! *  2. actposc  : actuator position at previous call
! *  3. actpospc : actuator position at previous time-step
! *  4. fspedc   : actuator speed at previous call
! *  5. fspedpc  : actuator speed at prev time-step
! *  6. ispedc   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedpc  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrevc  : total no. of stop/starts/reversals at previous call
! *  9. tssrevpc : total no. of stop/starts/reversals at prev time-step
! * 10. ttravc   : total distance travelled at previous call
! * 11. ttravpc  : total distance travelled at previous timestep
! * 12. dampposc : position of final control element at previous call
! * 13. damppospc: position of final control element at previous step time
! * 14. actposh  : actuator position at previous call
! * 15. actposph : actuator position at previous time-step
! * 16. fspedh   : actuator speed at previous call
! * 17. fspedph  : actuator speed at prev time-step
! * 18. ispedh   : +1=forward, -1=backward, 0=stopped at previous call
! * 19. ispedph  : +1=forward, -1=backward, 0=stopped at prev time-step
! * 20. tssrevh  : total no. of stop/starts/reversals at previous call
! * 21. tssrevph : total no. of stop/starts/reversals at prev time-step
! * 22. ttravh   : total distance travelled at previous call
! * 23. ttravph  : total distance travelled at previous timestep
! * 24. dampposh : position of final control element at previous call
! * 25. dampposph: position of final control element at previous step time

! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian, Xiwang LI, Jin Wen
!                        Drexel University
!                        Philip Haves
!                        Loughborough University of Technology
!                  
!
!   DATE:                June 16, 1994
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
! **********************************************************************
!
!   Updated to Fortran 90 Augest 2, 2012  Xiwang LI, Drexel University
!
! **********************************************************************
        
		subroutine type532(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=7,no=11,np=15,&
                                             ns=25
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat
		 
		real         :: rhoa = 1.2
        logical      :: con

! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-4.4941,-1.6325/),&
                                 blegg=(/0.1945,0.1181/)

! **** Loss coefficient of fully open damper
        real         :: kopenc = 2.0368, kopenh = 1.1149

! **** Assumed fractional flow leakage of closed damper
        real         :: fleakc = 0.0096, fleakh = 0.0149

! **** Assume damper is fully open at start
        real         :: startpos=1.0

        real         :: volumnomc,fareac,rboxc,ttranc,fspedminc,hysc,&
                        fspedpc,tssrevpc,ttravpc,damppospc,fflowc,&
                        fspedc,fspedbarc,actposc,tssrevc,ttravc,dampposc,&
                        aleggc,bleggc,rc,raddc,rtotc,pinc,wairc,pout,&
                        actpospc,ropenc,fflowsetc,fspeddemc,kpc,dpc
	    real         :: volumnomh,fareah,rboxh,ttranh,fspedminh,hysh,&
                        fspedph,tssrevph,ttravph,dampposph,fflowh,&
                        fspedh,fspedbarh,actposh,tssrevh,ttravh,dampposh,&
                        aleggh,bleggh,rh,raddh,rtoth,pinh,wairh,&
                        spedlim,clip,rdamper,dpqudlin,actposph,wout, &
						ropenh,fflowseth,fspeddemh,kph,dph,wqudlin,rdamper2
					
        integer      :: i,is,ispedpc,ispedc,ispedph,ispedh, location

! **** Distinguish hot & cold damper
        integer      :: ipc = 1, iph = 0

! **** Read in inputs
        pinc      = xin(1)
		pinh      = xin(2)
        pout      = xin(3)
		fflowsetc = xin(4)
        fflowseth = xin(5)
		wairc     = xin(6)
		wairh     = xin(7)

! **** Read and check parameters
        
        volumnomc = par_v(1)

        if (volumnomc<=0.0) then
            stop 'type 532: cold deck nominal flow rate must be greater than zero'
        endif
      
	    volumnomh = par_v(2)
        if (volumnomh<=0.0) then
            stop 'type 532: hot deck nominal flow rate must be greater than zero'
        endif

        fareac    = par_v(3)
		fareah    = par_v(4)
        rboxc     = par_v(5)
        rboxh     = par_v(6)
        ttranc    = par_v(7)
        ttranh    = par_v(8)
        fspedminc = par_v(9)
		fspedminh = par_v(10)
		hysc      = par_v(11)
		hysh      = par_v(12)
        kpc       = par_v(13)
		kph       = par_v(14)
		location  = nint(par_v(15))

! **** Initialize at beginning of simulation
        if (itime<=1) then 
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                saved_v(14) = startpos

                do is = 4,10,2
                    saved_v(is) = 0.0
                enddo
				do is = 16,22,2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
				saved_v(24) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif

! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actpospc  = saved_v(3)
        fspedpc   = saved_v(5)
        ispedpc   = saved_v(7)
        tssrevpc  = saved_v(9)
        ttravpc   = saved_v(11)
        damppospc = saved_v(13)
		actposph  = saved_v(15)
        fspedph   = saved_v(17)
        ispedph   = saved_v(19)
        tssrevph  = saved_v(21)
        ttravph   = saved_v(23)
        dampposph = saved_v(25)

! **** Limit control signal
        if (fflowsetc<0.0) fflowsetc = 0.0
        if (fflowsetc>1.0) fflowsetc = 1.0
        if (fflowseth<0.0) fflowseth = 0.0
        if (fflowseth>1.0) fflowseth = 1.0

! **** Measured fractional flow at end of time-step
        fflowc    = (wairc/rhoa)/volumnomc
		fflowh    = (wairh/rhoa)/volumnomh
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddemc = kpc*(fflowsetc-fflowc)
        fspedc    = spedlim(fspeddemc,fspedminc)

		fspeddemh = kph*(fflowseth-fflowh)
        fspedh    = spedlim(fspeddemh,fspedminh)
      
 ! **** Average speed over time-step
        fspedbarc = (fspedc+fspedpc)/2.0     
        fspedbarh = (fspedh+fspedph)/2.0     
! **** New actuator position
        actposc   = actpospc+fspedbarc*tstep/ttranc
        actposc   = clip(actposc,0.0,1.0)
		actposh   = actposph+fspedbarh*tstep/ttranh
        actposh   = clip(actposh,0.0,1.0)
!		if (actposh<=0.3) then
!		    actposh=actposh+0.2
!        endif

! **** Fault flag on
        if (location==1)then
		    if (CLGdamp==0) then
		        actposc = actposc
            elseif (CLGdamp==1) then
		        actposc = VCLGdamp
		    endif

		    if (HTGdamp==0) then
		        actposh = actposh
            elseif (HTGdamp==1) then
		        actposh = VHTGdamp
		    endif
		endif
! **** Fault flag off

! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fspedc>0.0) then
            ispedc = +1
        elseif (fspedc<0.0) then
            ispedc = -1
        else
            ispedc = 0
        endif

		if (fspedh>0.0) then
            ispedh = +1
        elseif (fspedh<0.0) then
            ispedh = -1
        else
            ispedh = 0
        endif

! **** Previous stops/starts/reversals
        if (ispedc/=ispedpc) then
            tssrevc = tssrevpc+1.0
        else
            tssrevc = tssrevpc
        endif

		 if (ispedh/=ispedph) then
            tssrevh = tssrevph+1.0
        else
            tssrevh = tssrevph
        endif
! **** Distance travelled
        ttravc = ttravpc+abs(actposc-actpospc)
		ttravh = ttravph+abs(actposh-actposph)

! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actposc,damppospc,hysc,dampposc)
		call hystrsis(actposh,dampposph,hysh,dampposh)

!		If (dampposh <=0.29) dampposh=0.49

! **** Resistance when fully open
        ropenc = 1.0e-3*kopenc/(2.0*rhoa*fareac*fareac)
		ropenh = 1.0e-3*kopenh/(2.0*rhoa*fareah*fareah)

! **** Resistance at position damppos
        aleggc = alegg(0)                 
        bleggc = blegg(0)                 
		aleggh = alegg(1)                 
        bleggh = blegg(1)                

        rc = rdamper2(dampposc,ropenc,fleakc,fareac,aleggc,bleggc,ipc)
		rh = rdamper2(dampposh,ropenh,fleakh,fareah,aleggh,bleggh,iph)
! **** Resistance of box, diffuser etc, excluding damper
!        raddc = pdopenc/(rhoa*volumnomc*rhoa*volumnomc) - ropenc
        rtotc = rc+rboxc
!		raddh = pdopenh/(rhoa*volumnomh*rhoa*volumnomh) - ropenh
        rtoth = rh+rboxh

! **** Calculate mass flow rate 
        dpc  = pinc-pout
		dph  = pinh-pout
		wairc= wqudlin(rtotc,dpc)
        wairh= wqudlin(rtoth,dph)
		wout=wairc+wairh

! **** Save time of current call
        saved_v(1)  = time

! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actposc
        saved_v(4)  = fspedc
        saved_v(6)  = float(ispedc)
        saved_v(8)  = tssrevc
        saved_v(10) = ttravc
        saved_v(12) = dampposc
        saved_v(14) = actposh
        saved_v(16) = fspedh
        saved_v(18) = float(ispedh)
        saved_v(20) = tssrevh
        saved_v(22) = ttravh
        saved_v(24) = dampposh

! **** Output
        yout(1) = wairc
		yout(2) = wairh
		yout(3) = wout
        yout(4) = dampposc
		yout(5) = dampposh
        yout(6) = fspedc
		yout(7) = fspedh
        yout(8) = tssrevc
		yout(9) = tssrevh
        yout(10) = ttravc
		yout(11)= ttravh

! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbarc==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type532

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent series fan powered
! *                 VAV box
! *
! * PURPOSE:        Calculate the inlet primary air pressure of a
! *                 motor-driven, pressure-independent fan powered
! *                 VAV box from the outlet pressure, the primary  
! *                 air mass flow rate and the demanded normalized
! *                 flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wprimair : primary air mass flow rate                     (kg/s)
! *  2. wout     : outlet air mass flow rate                      (kg/s)
! *  3. pout     : outlet pressure                                 (kPa)
! *  4. fflowset : demanded flowrate (normalized to nominal)       (0-1)
! *  5. N        : rotation speed                                (rev/s)
! *
! * OUTPUTS
! * =======
! *  1. pinprim  : inlet pressure                                  (kPa)
! *  2. damppos  : damper position (0=closed, 1=open)                (-)
! *  3. fsped    : fractional motor velocity (0-1)                   (-)
! *  4. tssrev   : number of stops/starts/reversals                  (-)
! *  5. ttrav    : total distance travelled by valve/damper          (-)
! *  6  wplenair : plenum air mass flow rate                      (kg/s)
! *  7. wout     : outlt air mass flow rate                       (kg/s)
! *  8. qa       : fluid stream heat addition rate                  (kW)
! *  9. power    : power consumption                                (kW)
! * 
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                    (m3/s)
! *  2. farea   : face area of damper(s)                            (m2)
! *  3. ttran   : travel time (0-90 deg)                             (s)
! *  4. fspedmin: minimum fractional motor speed                     (-)
! *  5. hys     : hysteresis                                         (-)
! *  6. kp      : controller gain (frac speed/frac error)            (-)
! *  7.         : coefficient of PN**0 in normalized head curve
! *  8.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  11.        : coefficient of PN**4 in normalized head curve
! *  12.        : coefficient of PN**0 in normalized head curve
! *  13.        : coefficient of PN**1 in normalized head curve
! * ...
! *  16.         : coefficient of PN**4 in normalized head curve
! *  17. D       : diameter                                          (-)
! *  18. WNCRITLO: normalised flow at lower bound of valid region    (-)
! *  19. WNCRITUP: normalised flow at upper bound of valid region    (-)
! *  20. rplen   : Resistance of plenum outlet               (0.001/k.m)
! *  21. routlet : Resistance of main outlet                 (0.001/k.m) 
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University
!
!   DATE:                July, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************


        subroutine type540(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=5,no=9,np=21,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

	    integer,parameter                 :: nifan=3,nofan=3,npfan=15,nsfan=5
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: volumnom,farea,ttran,fspedmin,hys,actposp,&
                        fspedp,tssrevp,ttravp,dampposp,fflow,fspeddem,&
                        fsped,fspedbar,actpos,tssrev,ttrav,damppos,ropen,&
                        alegg1,blegg1,r,radd,rtot,pin,wair,pout,fflowset,&
                        spedlim,clip,rdamper,dpqudlin,wqudlin,wprimair,&
						wout,N,po,wplenair,dp,pi,pinprim,rplen,routlet,qa,power
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        wprimair = xin(1)
        wout     = xin(2)
        pout     = xin(3)
	    fflowset = xin(4)
	    N        = xin(5)
! **** Read in par_vameters
        volumnom    = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 540: nominal flow rate must be greater than zero'
        endif
        farea       = par_v(2)
        ttran       = par_v(3)
        fspedmin    = par_v(4)
        hys         = par_v(5)
        kp          = par_v(6)
! **** Read in fan par_vameters
        par_vfan(1) = par_v(7)
	    par_vfan(2) = par_v(8)
	    par_vfan(3) = par_v(9)
	    par_vfan(4) = par_v(10)
        par_vfan(5) = par_v(11)
	    par_vfan(6) = par_v(12)
	    par_vfan(7) = par_v(13)
	    par_vfan(8) = par_v(14)
	    par_vfan(9) = par_v(15)
	    par_vfan(10)= par_v(16)
	    par_vfan(11)= par_v(17)
	    par_vfan(12)= 1
	    par_vfan(13)= par_v(18)
	    par_vfan(14)= par_v(19)
	    par_vfan(15)= 1
! **** Read in resistances
        rplen       = par_v(20)
        routlet     = par_v(21)	 
! **** Calculate fan outlet pressure knowing outlet resistance and outlet air mass flow rate	  
	    po = pout+dpqudlin(routlet,wout)
! **** Fan subroutine is called to calculate inlet fan pressure and heat addition rate to the fluid stream
        xinfan(1)   = wout
	    xinfan(2)   = po
	    xinfan(3)   = N
	  	  	   
        CALL TYPE355(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	    pi    = youtfan(1)
	    qa    = youtfan(2)
	    power = youtfan(3)
	  	   
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wprimair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
!        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
!        rtot = r+radd
! **** Pressure drop and primary air inlet pressure
        pinprim = pi+dpqudlin(r,wprimair)
! **** Plenum air mass flow rate
        dp = pout-pi
	    wplenair = wqudlin (rplen,dp)
! **** Outlet air mass flow rate
        wout = wprimair+wplenair
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = pinprim
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
	    yout(6) = wplenair
        yout(7) = wout
	    yout(8) = qa
	    yout(9) = power
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type540
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent series fan powered
! *                 VAV box - calculates flow
! *
! * PURPOSE:        Calculate the flow rate through a motor-driven,
! *                 pressure-independent  fan powered VAV box and 
! *                 flow rate of plenum from the primary and plenum
! *                 inlet pressures, the outlet pressure and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wout     : outlet air mass flow rate                      (kg/s)
! *  2. pout     : outlet pressure                                 (kPa)
! *  3. pinprim  : inlet primary air pressure                      (kPa)
! *  4. fflowset : demanded flowrate (normalized to nominal)       (0-1)
! *  5. N        : rotation speed                                (rev/s)
! *  
! *
! * OUTPUTS
! * =======
! *  1. wout     : outlt air mass flow rate                       (kg/s)
! *  2. damppos  : damper position (0=closed, 1=open)                (-)
! *  3. fsped    : fractional motor velocity (0-1)                   (-)
! *  4. tssrev   : number of stops/starts/reversals                  (-)
! *  5. ttrav    : total distance travelled by valve/damper          (-)
! *  6  wplenair : plenum air mass flow rate                      (kg/s)
! *  7. wprimair : primary air mass flow rate                     (kg/s)
! *  8. qa       : fluid stream heat addition rate                  (kW)
! *  9. power    : power consumption                                (kW)
! * 
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                    (m3/s)
! *  2. farea   : face area of damper(s)                            (m2)
! *  3. ttran   : travel time (0-90 deg)                             (s)
! *  4. fspedmin: minimum fractional motor speed                     (-)
! *  5. hys     : hysteresis                                         (-)
! *  6. kp      : controller gain (frac speed/frac error)            (-)
! *  7.         : coefficient of PN**0 in normalized head curve
! *  8.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  11.        : coefficient of PN**4 in normalized head curve
! *  12.        : coefficient of PN**0 in normalized head curve
! *  13.        : coefficient of PN**1 in normalized head curve
! * ...
! *  16.         : coefficient of PN**4 in normalized head curve
! *  17. D       : diameter                                          (-)
! *  18. WNCRITLO: normalised flow at lower bound of valid region    (-)
! *  19. WNCRITUP: normalised flow at upper bound of valid region    (-)
! *  20. rplen   : Resistance of plenum outlet               (0.001/k.m)
! *  21. routlet : Resistance of main outlet                 (0.001/k.m) 
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University
!
!   DATE:                July, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************


        subroutine type541(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=5,no=9,np=21,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

	    integer,parameter                 :: nifan=3,nofan=3,npfan=15,nsfan=5
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: volumnom,farea,ttran,fspedmin,hys,actposp,&
                        fspedp,tssrevp,ttravp,dampposp,fflow,fspeddem,&
                        fsped,fspedbar,actpos,tssrev,ttrav,damppos,ropen,&
                        alegg1,blegg1,r,radd,rtot,pin,wair,pout,fflowset,&
                        spedlim,clip,rdamper,dpqudlin,wqudlin,wout,pinprim,&
						N,rplen,routlet,po,pi,qa,power,dpplen,dpprim,&
						wplenair,wprimair
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        wout     = xin(1)
        pout     = xin(2)
	    pinprim  = xin(3)
	    fflowset = xin(4)
	    N        = xin(5)

! **** Read in par_vameters
        volumnom    = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 541: nominal flow rate must be greater than zero'
        endif
        farea       = par_v(2)
        ttran       = par_v(3)
        fspedmin    = par_v(4)
        hys         = par_v(5)
        kp          = par_v(6)
! **** Read in fan par_vameters
        par_vfan(1) = par_v(7)
	    par_vfan(2) = par_v(8)
	    par_vfan(3) = par_v(9)
	    par_vfan(4) = par_v(10)
	    par_vfan(5) = par_v(11)
	    par_vfan(6) = par_v(12)
	    par_vfan(7) = par_v(13)
	    par_vfan(8) = par_v(14)
	    par_vfan(9) = par_v(15)
	    par_vfan(10)= par_v(16)
	    par_vfan(11)= par_v(17)
	    par_vfan(12)= 1
	    par_vfan(13)= par_v(18)
	    par_vfan(14)= par_v(19)
	    par_vfan(15)= 1
! **** Read in resistances
        rplen       = par_v(20)
        routlet     = par_v(21)	 
! **** Calculate fan outlet pressure knowing outlet resistance and outlet air mass flow rate	  
	    po = pout+dpqudlin(routlet,wout)
! **** Fan subroutine is called to calculate inlet fan pressure and heat addition rate to the fluid stream
        xinfan(1)   = wout
	    xinfan(2)   = po
	    xinfan(3)   = N   
        CALL TYPE355(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	    pi    = youtfan(1)
	    qa    = youtfan(2)
	    power = youtfan(3)
! **** Calculate plenum mass flow rate 	
	    dpplen   = pout-pi
        wplenair = wqudlin(rplen,dpplen)
! **** Calculate primary mass flow rate 	
	    wprimair = wout-wplenair
	  	   
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wprimair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
!        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
!        rtot = r+radd
! **** Pressure drop and primary air mass flow rate
        dpprim   = pinprim-pi
        wprimair = wqudlin(r,dpprim)
! **** Outlet air mass flow rate
        wout     = wprimair+wplenair
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos

! **** Output
        yout(1) = wout
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
	    yout(6) = wplenair
        yout(7) = wprimair
	    yout(8) = qa
	    yout(9) = power
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type541
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent parallel fan powered
! *                 VAV box
! *
! * PURPOSE:        Calculate the inlet primary air pressure of a
! *                 motor-driven, pressure-independent fan powered
! *                 VAV box from the outlet pressure, the primary  
! *                 air mass flow rate and the demanded normalized
! *                 flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wprimair : primary air mass flow rate                     (kg/s)
! *  2. wout     : outlet air mass flow rate                      (kg/s)
! *  3. pout     : outlet pressure                                 (kPa)
! *  4. fflowset : demanded flowrate (normalized to nominal)       (0-1)
! *  5. N        : rotation speed                                (rev/s)
! *
! * OUTPUTS
! * =======
! *  1. pinprim  : inlet pressure                                  (kPa)
! *  2. damppos  : damper position (0=closed, 1=open)                (-)
! *  3. fsped    : fractional motor velocity (0-1)                   (-)
! *  4. tssrev   : number of stops/starts/reversals                  (-)
! *  5. ttrav    : total distance travelled by valve/damper          (-)
! *  6  wplenair : plenum air mass flow rate                      (kg/s)
! *  7. wout     : outlt air mass flow rate                       (kg/s)
! *  8. qa       : fluid stream heat addition rate                  (kW)
! *  9. power    : power consumption                                (kW)
! * 
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                    (m3/s)
! *  2. farea   : face area of damper(s)                            (m2)
! *  3. ttran   : travel time (0-90 deg)                             (s)
! *  4. fspedmin: minimum fractional motor speed                     (-)
! *  5. hys     : hysteresis                                         (-)
! *  6. kp      : controller gain (frac speed/frac error)            (-)
! *  7.         : coefficient of PN**0 in normalized head curve
! *  8.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  11.        : coefficient of PN**4 in normalized head curve
! *  12.        : coefficient of PN**0 in normalized head curve
! *  13.        : coefficient of PN**1 in normalized head curve
! * ...
! *  16.         : coefficient of PN**4 in normalized head curve
! *  17. D       : diameter                                          (-)
! *  18. WNCRITLO: normalised flow at lower bound of valid region    (-)
! *  19. WNCRITUP: normalised flow at upper bound of valid region    (-)
! *  20. rplen   : Resistance of plenum outlet               (0.001/k.m)
! *  21. routlet : Resistance of main outlet                 (0.001/k.m) 
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University
!
!   DATE:                July, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************


        subroutine type542(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=5,no=9,np=21,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

	    integer,parameter                 :: nifan=3,nofan=3,npfan=15,nsfan=5
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: volumnom,farea,ttran,fspedmin,hys,actposp,&
                        fspedp,tssrevp,ttravp,dampposp,fflow,fspeddem,&
                        fsped,fspedbar,actpos,tssrev,ttrav,damppos,ropen,&
                        alegg1,blegg1,r,radd,rtot,pin,wair,pout,fflowset,&
                        spedlim,clip,rdamper,dpqudlin,wqudlin,wprimair,&
						wout,N,po,wplenair,dp,pi,pinprim,rplen,routlet,qa,power
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        wprimair = xin(1)
        wout     = xin(2)
        pout     = xin(3)
	    fflowset = xin(4)
	    N        = xin(5)
! **** Read in par_vameters
        volumnom    = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 540: nominal flow rate must be greater than zero'
        endif
        farea       = par_v(2)
        ttran       = par_v(3)
        fspedmin    = par_v(4)
        hys         = par_v(5)
        kp          = par_v(6)
! **** Read in fan par_vameters
        par_vfan(1) = par_v(7)
	    par_vfan(2) = par_v(8)
	    par_vfan(3) = par_v(9)
	    par_vfan(4) = par_v(10)
        par_vfan(5) = par_v(11)
	    par_vfan(6) = par_v(12)
	    par_vfan(7) = par_v(13)
	    par_vfan(8) = par_v(14)
	    par_vfan(9) = par_v(15)
	    par_vfan(10)= par_v(16)
	    par_vfan(11)= par_v(17)
	    par_vfan(12)= 1
	    par_vfan(13)= par_v(18)
	    par_vfan(14)= par_v(19)
	    par_vfan(15)= 1
! **** Read in resistances
        rplen       = par_v(20)
        routlet     = par_v(21)	 
! **** Calculate fan outlet pressure knowing outlet resistance and outlet air mass flow rate	  
	    po = pout+dpqudlin(routlet,wout)
! **** Calculate plenum air mass flow rate
        wplenair = wout-wprimair
! **** Fan subroutine is called to calculate inlet fan pressure and heat addition rate to the plenum air stream
        xinfan(1)   = wplenair
	    xinfan(2)   = po
	    xinfan(3)   = N
	  	  	   
        CALL TYPE355(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	    pi    = youtfan(1)
	    qa    = youtfan(2)
	    power = youtfan(3)
! **** Calculate plenum air mass flow rate
        dp       = pout-pi
        wplenair = wqudlin (rplen,dp)
	  	   
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wprimair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)
! **** Resistance of box, diffuser etc, excluding damper
!        radd = pdopen/(rhoa*volumnom*rhoa*volumnom) - ropen
!        rtot = r+radd
! **** Pressure drop and primary air inlet pressure
        pinprim = po+dpqudlin(r,wprimair)
! **** Outlet air mass flow rate
        wout = wprimair+wplenair
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos
! **** Output
        yout(1) = pinprim
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
	    yout(6) = wplenair
        yout(7) = wout
	    yout(8) = qa
	    yout(9) = power
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type542
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:     Motorized pressure-independent parallel fan powered
! *                 VAV box - calculates flow
! *
! * PURPOSE:        Calculate the flow rate through a motor-driven,
! *                 pressure-independent  fan powered VAV box and 
! *                 flow rate of plenum from the primary and plenum
! *                 inlet pressures, the outlet pressure and the demanded
! *                 normalized flow rate.
! *
! *********************************************************************
! * INPUTS
! * ======
! *  1. wprimair : outlet air mass flow rate                      (kg/s)
! *  2. pout     : outlet pressure                                 (kPa)
! *  3. pinprim  : inlet primary air pressure                      (kPa)
! *  4. fflowset : demanded flowrate (normalized to nominal)       (0-1)
! *  5. N        : rotation speed                                (rev/s)
! *  
! *
! * OUTPUTS
! * =======
! *  1. wout     : outlt air mass flow rate                       (kg/s)
! *  2. damppos  : damper position (0=closed, 1=open)                (-)
! *  3. fsped    : fractional motor velocity (0-1)                   (-)
! *  4. tssrev   : number of stops/starts/reversals                  (-)
! *  5. ttrav    : total distance travelled by valve/damper          (-)
! *  6  wplenair : plenum air mass flow rate                      (kg/s)
! *  7. wprimair : primary air mass flow rate                     (kg/s)
! *  8. qa       : fluid stream heat addition rate                  (kW)
! *  9. power    : power consumption                                (kW)
! * 
! * PARAMETERS
! * ==========
! *  1. volumnom: nominal volumetric flow rate                    (m3/s)
! *  2. farea   : face area of damper(s)                            (m2)
! *  3. ttran   : travel time (0-90 deg)                             (s)
! *  4. fspedmin: minimum fractional motor speed                     (-)
! *  5. hys     : hysteresis                                         (-)
! *  6. kp      : controller gain (frac speed/frac error)            (-)
! *  7.         : coefficient of PN**0 in normalized head curve
! *  8.         : coefficient of PN**1 in normalized head curve
! *  ...
! *  11.        : coefficient of PN**4 in normalized head curve
! *  12.        : coefficient of PN**0 in normalized head curve
! *  13.        : coefficient of PN**1 in normalized head curve
! * ...
! *  16.         : coefficient of PN**4 in normalized head curve
! *  17. D       : diameter                                          (-)
! *  18. WNCRITLO: normalised flow at lower bound of valid region    (-)
! *  19. WNCRITUP: normalised flow at upper bound of valid region    (-)
! *  20. rplen   : Resistance of plenum outlet               (0.001/k.m)
! *  21. routlet : Resistance of main outlet                 (0.001/k.m) 
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call for rate limit
! *  2. actpos  : actuator position at previous call
! *  3. actposp : actuator position at previous time-step
! *  4. fsped   : actuator speed at previous call
! *  5. fspedp  : actuator speed at prev time-step
! *  6. isped   : +1=forward, -1=backward, 0=stopped at previous call
! *  7. ispedp  : +1=forward, -1=backward, 0=stopped at prev time-step
! *  8. tssrev  : total no. of stop/starts/reversals at previous call
! *  9. tssrevp : total no. of stop/starts/reversals at prev time-step
! * 10. ttrav   : total distance travelled at previous call
! * 11. ttravp  : total distance travelled at previous timestep
! * 12. damppos : position of final control element at previous call
! * 13. dampposp: position of final control element at previous step time
! *
! *********************************************************************
!
!   MAJOR RESTRICTIONS:
!
!   DEVELOPER:           Shokouh Pourarian, Jin Wen
!                        Drexel University
!
!   DATE:                July, 2012
!
!   INCLUDE FILES:
!   SUBROUTINES CALLED:
!   FUNCTIONS  CALLED:   HYSTER2
!
!   REVISION HISTORY:    None
!
!   REFERENCE:           Haves, P., Component-Based Modelling of VAV
!                        Systems, Proc. System Simulation in Buildings
!                        '94, Liege, Belgium, December 1994
!
! **********************************************************************


        subroutine type543(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ndtdt=0,ndiffeq=0,&
                                             ni=5,no=9,np=21,&
                                             ns=13
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

	    integer,parameter                 :: nifan=3,nofan=3,npfan=15,nsfan=5
        real,dimension(nifan)             :: xinfan
        real,dimension(nofan)             :: youtfan                     
        real,dimension(npfan)             :: par_vfan                    
        real,dimension(nsfan)             :: saved_vfan                  
        integer,dimension(nofan)          :: iostatfan

        real         :: kp
        logical      :: con
        real         :: rhoa=1.2
! **** Damper coefficients (ln(k)=a+b*theta) from legg
        real,dimension(0:1)   :: alegg=(/-1.51,-1.51/),&
                                 blegg=(/0.105,0.0842/)
! **** Loss coefficient of fully open damper
        real         :: kopen=0.2
! **** Assumed fractional flow leakage of closed damper
        real         :: fleak=1.0e-2
! **** Assume opposed or single blade damper
        integer      :: ipx=0
! **** Assume damper is fully open at start
        real         :: startpos=1.0
        real         :: volumnom,farea,ttran,fspedmin,hys,actposp,&
                        fspedp,tssrevp,ttravp,dampposp,fflow,fspeddem,&
                        fsped,fspedbar,actpos,tssrev,ttrav,damppos,ropen,&
                        alegg1,blegg1,r,radd,rtot,pin,wair,pout,fflowset,&
                        spedlim,clip,rdamper,dpqudlin,wqudlin,wout,pinprim,&
						N,rplen,routlet,po,pi,qa,power,dpplen,dp,&
						wplenair,wprimair
        integer      :: i,is,ispedp,isped

! **** Read in inputs
        wprimair = xin(1)
        pout     = xin(2)
	    pinprim  = xin(3)
	    fflowset = xin(4)
	    N        = xin(5)

! **** Read in par_vameters
        volumnom    = par_v(1)
        if (volumnom<=0.0) then
            stop 'type 541: nominal flow rate must be greater than zero'
        endif
        farea       = par_v(2)
        ttran       = par_v(3)
        fspedmin    = par_v(4)
        hys         = par_v(5)
        kp          = par_v(6)
! **** Read in fan par_vameters
        par_vfan(1) = par_v(7)
	    par_vfan(2) = par_v(8)
	    par_vfan(3) = par_v(9)
	    par_vfan(4) = par_v(10)
	    par_vfan(5) = par_v(11)
	    par_vfan(6) = par_v(12)
	    par_vfan(7) = par_v(13)
	    par_vfan(8) = par_v(14)
	    par_vfan(9) = par_v(15)
	    par_vfan(10)= par_v(16)
	    par_vfan(11)= par_v(17)
	    par_vfan(12)= 1
	    par_vfan(13)= par_v(18)
	    par_vfan(14)= par_v(19)
	    par_vfan(15)= 1
! **** Read in resistances
        rplen       = par_v(20)
        routlet     = par_v(21)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
                saved_v(2)  = startpos
                do is = 4,(ns-3),2
                    saved_v(is) = 0.0
                enddo
                saved_v(12) = startpos
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update previous sample instant values
            do is=2,(ns-1),2
                saved_v(is+1) = saved_v(is)
            enddo
        endif
! **** Determine curent position based on situation at previous step time
! **** Update previous values
        actposp  = saved_v(3)
        fspedp   = saved_v(5)
        ispedp   = saved_v(7)
        tssrevp  = saved_v(9)
        ttravp   = saved_v(11)
        dampposp = saved_v(13)
! **** Limit control signal
        if (fflowset<0.0) fflowset = 0.0
        if (fflowset>1.0) fflowset = 1.0
! **** Measured fractional flow at end of time-step
        fflow    = (wprimair/rhoa)/volumnom
! **** Fractional speed of motor at end of time-step - proportional to
! **** control error and limited to range fspedmin - 1.0 or zero
        fspeddem = kp*(fflowset-fflow)
        fsped    = spedlim(fspeddem,fspedmin)
! **** Average speed over time-step
        fspedbar = (fsped+fspedp)/2.0
! **** New actuator position
        actpos   = actposp+fspedbar*tstep/ttran
        actpos   = clip(actpos,0.0,1.0)
! **** Operational statistics
! **** Stops/starts/reversals
! **** Current motor direction
        if (fsped>0.0) then
            isped = +1
        elseif (fsped<0.0) then
            isped = -1
        else
            isped = 0
        endif
! **** Previous stops/starts/reversals
        if (isped/=ispedp) then
            tssrev = tssrevp+1.0
        else
            tssrev = tssrevp
        endif
! **** Distance travelled
        ttrav = ttravp+abs(actpos-actposp)
! ****  hysteresis (no correction for reduction in range)
        call hystrsis(actpos,dampposp,hys,damppos)
! **** Damper resistance
! **** Resistance when fully open
        ropen = 1.0e-3*kopen/(2.0*rhoa*farea*farea)
! **** Resistance at position damppos
        alegg1 = alegg(0)                 ! added 12/23/99
        blegg1 = blegg(0)                 ! added 12/23/99
        r = rdamper(damppos,ropen,fleak,farea,alegg1,blegg1,ipx)			 
! **** Calculate fan outlet pressure knowing primary air inlet pressure and air mass flow rate	  
	    po = pinprim-dpqudlin(r,wprimair)
! **** Calculate outlet air mass flow rate 	
	    dp   = po-pout
        wout = wqudlin(routlet,dp)
! **** Calculate plenum air mass flow rate 
        wplenair = wout-wprimair
! **** Fan subroutine is called to calculate inlet fan pressure and heat addition rate to the fluid stream
        xinfan(1)   = wplenair
	    xinfan(2)   = po
	    xinfan(3)   = N   
        CALL TYPE355(xinfan,youtfan,par_vfan,saved_vfan,iostatfan)
	    pi    = youtfan(1)
	    qa    = youtfan(2)
	    power = youtfan(3)
! **** Calculate plenum mass flow rate 	
	    dpplen   = pout-pi
        wplenair = wqudlin(rplen,dpplen)
! **** Calculate primary mass flow rate 	
	    wprimair = wout-wplenair
! **** Save time of current call
        saved_v(1)  = time
! **** Save provisional values to be used in updating at next step time
        saved_v(2)  = actpos
        saved_v(4)  = fsped
        saved_v(6)  = float(isped)
        saved_v(8)  = tssrev
        saved_v(10) = ttrav
        saved_v(12) = damppos

! **** Output
        yout(1) = wout
        yout(2) = damppos
        yout(3) = fsped
        yout(4) = tssrev
        yout(5) = ttrav
	    yout(6) = wplenair
        yout(7) = wprimair
	    yout(8) = qa
	    yout(9) = power
! **** Determine whether to allow freezing
! **** Freezing allowed if motor stopped and demanded flow no changing
        con = (iostat(3)<-1).or.(iostat(3)==2)
        if (fspedbar==0.0.and.con) then
            do i=1,no
                iostat(i) = 1
            enddo
        else
            do i=1,no
                iostat(i) = 1
            enddo
        endif
! **** Return
        return
        end subroutine type543
